/*******************************************************************
 *
 * xpath-test
 *
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 *
 ******************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h> 
#include <time.h> 

#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

static const char usage[] = "[-childs] [-here <Id>] [-count <N>] [-print] <doc-file> <xpath-file>";


int xpathTest(const char *docFile, const char *xpathFile, const char *here, int childs);
xmlXPathObjectPtr evalXPath(xmlDocPtr parentDoc, const char* filename, const char *here);
void printNodesSet(xmlNodeSetPtr nodes);
xmlNodePtr findNodeById(const xmlNodePtr parent, const xmlChar *id);
void XPathHereFunction(xmlXPathParserContextPtr ctxt, int nargs);
void addChilds(xmlNodeSetPtr nodes, xmlNodePtr cur);

clock_t total_time = 0;
int print = 0;

int main(int argc, char **argv) {
    int pos;
    int ret;
    int childs = 0;
    int count = 1;
    char *here = NULL;
    int i;
    
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION

    pos = 1;
    while((pos < argc) && (argv[pos][0] == '-')) {
	if(strcmp(argv[pos], "-childs") == 0) {
	    childs = 1;
	    ++pos;
	} else if(strcmp(argv[pos], "-print") == 0) {
	    print = 1;
	    ++pos;
	} else if((strcmp(argv[pos], "-here") == 0) && ((pos + 1) < argc)) {
	    here = argv[++pos];
	    ++pos;
	} else if((strcmp(argv[pos], "-count") == 0) && ((pos + 1) < argc)) {
	    sscanf(argv[++pos], "%d", &count);
	    ++pos;
	} else {
	    fprintf(stderr, "Error: bad option \"%s\".\n", argv[pos]);   
	    fprintf(stderr, "Usage: %s %s\n", argv[0], usage);
	    return(1);
	}
    }
    
    if(pos + 1 >= argc) {
	fprintf(stderr, "Error: not enough arguments.\n");   
	fprintf(stderr, "Usage: %s %s\n", argv[0], usage);
	return(1);
    }
    
    for(i = 0; i < count; ++i) {
	ret = xpathTest(argv[pos], argv[pos + 1], here, childs);
	if(ret < 0) {
	    fprintf(stderr, "Usage: %s %s\n", argv[0], usage);
	    return(1);
	}
    }

    fprintf(stdout, "XPath-Test: Executed %d tests in %ld msec\n", count, total_time / (CLOCKS_PER_SEC / 1000));    
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    return(0);            
}


int xpathTest(const char *docFile, const char *xpathFile, const char *here, int childs) {
    xmlDocPtr doc = NULL;
    xmlXPathObjectPtr xpath = NULL;
    clock_t start_time;
    
    /*
     * build an XML tree from a the file; we need to add default
     * attributes and resolve all character and entities references
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    doc = xmlParseFile(docFile);
    if (doc == NULL) {
	fprintf(stderr, "Error: unable to parse file \"%s\"\n", docFile);
	return(-1);
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", docFile);
	xmlFreeDoc(doc);
	return(-1);
    }
    
    xpath = evalXPath(doc, xpathFile, here);
    if(xpath == NULL) {
        fprintf(stderr,"Error: xpath expression evaluation failed \"%s\"\n", xpathFile);
	xmlFreeDoc(doc);
	return(-1);
    }

    if(childs) {
	/* add child nodes */
	start_time = clock();
	addChilds(xpath->nodesetval, NULL); 
	total_time += clock() - start_time; 
    }

    if(print) {
	printNodesSet(xpath->nodesetval);
	print = 0;
    }
    
    xmlXPathFreeObject(xpath);  
    xmlFreeDoc(doc);
    return(0);
}

xmlXPathObjectPtr evalXPath(xmlDocPtr parentDoc, const char* filename, const char *here) {
    xmlXPathObjectPtr xpath; 
    xmlDocPtr doc;
    xmlChar *expr;
    xmlXPathContextPtr ctx; 
    xmlNodePtr node;
    xmlNsPtr ns;
    clock_t start_time;
    
    /*
     * load XPath expr as a file
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    doc = xmlParseFile(filename);
    if (doc == NULL) {
	fprintf(stderr, "Error: unable to parse file \"%s\"\n", filename);
	return(NULL);
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", filename);
	xmlFreeDoc(doc);
	return(NULL);
    }

    node = doc->children;
    while(node != NULL && !xmlStrEqual(node->name, (const xmlChar *)"XPath")) {
	node = node->next;
    }
    
    if(node == NULL) {   
        fprintf(stderr,"Error: XPath element expected in the file  \"%s\"\n", filename);
	xmlFreeDoc(doc);
	return(NULL);
    }

    expr = xmlNodeGetContent(node);
    if(expr == NULL) {
        fprintf(stderr,"Error: XPath content element is NULL \"%s\"\n", filename);
	xmlFreeDoc(doc);
	return(NULL);
    }

    ctx = xmlXPathNewContext(parentDoc);
    if(ctx == NULL) {
        fprintf(stderr,"Error: unable to create new context\n");
        xmlFree(expr); 
        xmlFreeDoc(doc); 
        return(NULL);
    }

    /*
     * Register namespaces
     */
    ns = node->nsDef;
    while(ns != NULL) {
	if(xmlXPathRegisterNs(ctx, ns->prefix, ns->href) != 0) {
	    fprintf(stderr,"Error: unable to register NS with prefix=\"%s\" and href=\"%s\"\n", ns->prefix, ns->href);
    	    xmlFree(expr); 
	    xmlXPathFreeContext(ctx); 
	    xmlFreeDoc(doc); 
	    return(NULL);
	}
	ns = ns->next;
    }

    if(here != NULL) {
	/* set here() node */
	xmlNodePtr ptr;
	
	ptr = findNodeById(xmlDocGetRootElement(parentDoc), here);
	xmlXPathRegisterFunc(ctx, (xmlChar *)"here", XPathHereFunction);
	ctx->here = ptr;
	ctx->xptr = 1;
    }

    /*  
     * Evaluate xpath and time it
     */
    start_time = clock(); 
    xpath = xmlXPathEvalExpression(expr, ctx);
    total_time += clock() - start_time; 

    if(xpath == NULL) {
        fprintf(stderr,"Error: unable to evaluate xpath expression\n");
    	xmlFree(expr); 
        xmlXPathFreeContext(ctx); 
        xmlFreeDoc(doc); 
        return(NULL);
    }

    xmlFree(expr); 
    xmlXPathFreeContext(ctx); 
    xmlFreeDoc(doc); 
    return(xpath);
}

void printNodesSet(xmlNodeSetPtr nodes) {
    xmlNodePtr cur;
    int i;
    
    if(nodes == NULL ){ 
	fprintf(stderr, "Error: no nodes set defined\n");
	return;
    }
    
    fprintf(stderr, "Nodes Set:\n-----\n");
    for(i = 0; i < nodes->nodeNr; ++i) {
	cur = nodes->nodeTab[i];    
	fprintf(stderr, "node \"%s\": type %d\n", cur->name, cur->type);
    }
}

xmlNodePtr findNodeById(const xmlNodePtr parent, const xmlChar *id) {
    xmlNodePtr cur;
    xmlChar* attr;
    xmlNodePtr ret;
    
    if(parent == NULL) {
	return(NULL);
    }
    
    attr = xmlGetProp(parent, BAD_CAST "Id");
    if(xmlStrEqual(id, attr)) {
	xmlFree(attr);
	return(parent);
    }
    xmlFree(attr);

    cur = parent->children;
    while(cur != NULL) {
        if(cur->type == XML_ELEMENT_NODE) {	    
	    ret = findNodeById(cur, id);
	    if(ret != NULL) {
	        return(ret);	    
	    }
	}
	cur = cur->next;
    }
    return(NULL);
}

/** 
 * see xmlXPtrHereFunction() in xpointer.c. the only change is that 
 * we return NodeSet instead of NodeInterval
 */
void  XPathHereFunction(xmlXPathParserContextPtr ctxt, int nargs) {
    CHECK_ARITY(0);

    if (ctxt->context->here == NULL)
	XP_ERROR(XPTR_SYNTAX_ERROR);
    
    valuePush(ctxt, xmlXPathNewNodeSet(ctxt->context->here));
}

void addChilds(xmlNodeSetPtr nodes, xmlNodePtr cur) {
    if(cur == NULL) {
	int i, n;
	
	n = nodes->nodeNr;
	for(i = 0; i < n; ++i) { 
	    addChilds(nodes, nodes->nodeTab[i]);
	}
    } else {
	xmlXPathNodeSetAddUnique(nodes, cur);
	
	cur = cur->children;
	while(cur != NULL) {
	    addChilds(nodes, cur); 
	    cur = cur->next;
	}
    }
}


