/** 
 * XML Security examples
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/err.h>
#include <openssl/rand.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <xmlsec/keysmgmt.h> 
#include <xmlsec/algorithms.h>
#include <xmlsec/xmldsig.h>



int main(int argc, char **argv) {
    xmlSecSimpleKeyMngrPtr keyMgr = NULL; 
    xmlDocPtr doc = NULL;
    xmlDSigCtxPtr dsigCtx = NULL;
    xmlChar* result;
    int ret = -1;
    int rnd_seed = 0;
    int len; 
        
    if(argc < 2) {
	fprintf(stderr, "Error: missed required parameter. Usage: %s <keys-file> <xml-file>\n", argv[0]);
	return(1);
    }
    
    /** 
     * Init OpenSSL
     */    
    while (RAND_status() != 1) {
	RAND_seed(&rnd_seed, sizeof(rnd_seed));
    }
    
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION

    /** 
     * load keys 
     */
    keyMgr = xmlSecSimpleKeyMngrCreate();
    if(keyMgr == NULL) {
	fprintf(stderr, "Error: failed to create keys manager\n");
	goto done;	
    }

    if(xmlSecSimpleKeyMngrLoad(keyMgr, argv[1]) < 0) {
	fprintf(stderr, "Error: failed to load keys from \"%s\"\n", argv[1]);
	goto done;
    }

    /*xmlDocGetRootElement(doc)
     * build an XML tree from a the file; we need to add default
     * attributes and resolve all character and entities references
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    /** 
     * Load doc 
     */
    doc = xmlParseFile(argv[2]);
    if (doc == NULL) {
	fprintf(stderr, "Error	: unable to parse file \"%s\"\n", argv[2]);
	goto done;
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", argv[2]);
	goto done;
    }
    
    /*
     * Add Data to the document
     */
    if(xmlNewChild(xmlDocGetRootElement(doc), NULL, "Something", 
		  "Some important data") == NULL) {
        fprintf(stderr,"Error: failed to add data\n");
	goto done;
    }
    
    /**
     * Create Signature Context 
     */
    dsigCtx = xmlDSigCtxCreate(keyMgr, NULL);
    if(dsigCtx == NULL) {
    	fprintf(stderr,"Error: failed to create dsig context\n");
	goto done; 
    }

            
    /**
     * Sign It!
     */ 
    if(xmlDSigGenerate(dsigCtx, doc, NULL, NULL) < 0) {
    	fprintf(stderr,"Error: signature failed\n");
	goto done; 
    }     
    
    /*
     * Print out result document
     */
    xmlDocDumpMemoryEnc(doc, &result, &len, NULL);
    if(result == NULL) {
	fprintf(stderr,"Error: failed to dump document to memory\n");
	goto done;
    }
    fwrite(result, len, 1, stdout);
    xmlFree(result);
    
done:
    /*
     * Cleanup
     */
    if(dsigCtx != NULL) { 
	xmlDSigCtxDestroy(dsigCtx);
    }
    if(doc != NULL) {
	xmlFreeDoc(doc); 
    }
    
    if(keyMgr != NULL) {
	xmlSecKeyMngrDestroy(keyMgr);
    }
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    
    /* 
     * Shutdown OpenSSL
     */
    RAND_cleanup();
    ERR_clear_error();

    return((ret >= 0) ? 0 : 1);
}

