/** 
 * XML Security examples
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/err.h>
#include <openssl/rand.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <xmlsec/keysmgmt.h> 
#include <xmlsec/algorithms.h>
#include <xmlsec/xmldsig.h>
#include <xmlsec/xpath.h>


char *namespaces[] = { 
    "test1", "http://www.aleksey.com/test1",   
    "test2", "http://www.aleksey.com/test2",   
    NULL, NULL};
xmlSecXPathTransformData textOnly = { "self::text()",  namespaces } ;


int main(int argc, char **argv) {
    xmlSecSimpleKeyMngrPtr keyMgr = NULL; 
    xmlDocPtr doc = NULL;
    xmlDSigCtxPtr dsigCtx = NULL;
    xmlNodePtr signatureNode;
    xmlNodePtr referenceNode;
    int ret = -1;
    int rnd_seed = 0;
    xmlChar *result;
    int len; 
        
    if(argc < 2) {
	fprintf(stderr, "Error: missed required parameter. Usage: %s <keys-file> <xml-file>\n", argv[0]);
	return(1);
    }
    
    /** 
     * Init OpenSSL
     */    
    while (RAND_status() != 1) {
	RAND_seed(&rnd_seed, sizeof(rnd_seed));
    }
    
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION

    /** 
     * load keys 
     */
    keyMgr = xmlSecSimpleKeyMngrCreate();
    if(keyMgr == NULL) {
	fprintf(stderr, "Error: failed to create keys manager\n");
	goto done;	
    }

    if(xmlSecSimpleKeyMngrLoad(keyMgr, argv[1]) < 0) {
	fprintf(stderr, "Error: failed to load keys from \"%s\"\n", argv[1]);
	goto done;
    }

    /*
     * build an XML tree from a the file; we need to add default
     * attributes and resolve all character and entities references
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    /** 
     * Load doc 
     */
    doc = xmlParseFile(argv[2]);
    if (doc == NULL) {
	fprintf(stderr, "Error	: unable to parse file \"%s\"\n", argv[2]);
	goto done;
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", argv[2]);
	goto done;
    }

    /**
     * Create Signature Context 
     */
    dsigCtx = xmlDSigCtxCreate(keyMgr, NULL);
    if(dsigCtx == NULL) {
    	fprintf(stderr,"Error: failed to create dsig context\n");
	goto done; 
    }

    /**
     * Create Signature node 
     */
    signatureNode = xmlDSigCreate(dsigCtx, doc, 
				xmlSecC14NInclusive, NULL,
				xmlSecSignDsaSha1, NULL);
    if(signatureNode == NULL) {
    	fprintf(stderr,"Error: failed to create signature\n");
	goto done; 
    }

    /**
     * Add the signature to the end of the document
     */    
    if(xmlAddChild(xmlDocGetRootElement(doc), signatureNode) == NULL) {
    	fprintf(stderr,"Error: failed to add signature\n");
	goto done; 
    }

    /** 
     * Create Reference node
     */
    referenceNode = xmlDSigReferenceAdd(dsigCtx, doc, signatureNode,
					"#xpointer(id('SignedInfo'))",
					xmlSecDigestSha1, NULL);
    if(referenceNode == NULL) {
    	fprintf(stderr,"Error: failed to add reference\n");
	goto done; 
    }

    /**
     * Add XPath Transform to select text
     */
    if(xmlDSigTransformAdd(dsigCtx, doc, referenceNode, xmlSecTransformXpath, 
					&textOnly) == NULL) {
    	fprintf(stderr,"Error: failed to add trasnform\n");
	goto done; 	
    }

    /**
     * Sign It!
     */ 
    if(xmlDSigGenerate(dsigCtx, doc, signatureNode, NULL) < 0) {
    	fprintf(stderr,"Error: signature failed\n");
	goto done; 
    }     
    
    /*
     * Print out result document
     */
    xmlDocDumpMemoryEnc(doc, &result, &len, NULL);
    if(result == NULL) {
	fprintf(stderr,"Error: failed to dump document to memory\n");
	goto done;
    }
    fwrite(result, len, 1, stdout);
    xmlFree(result);
    
done:
    /*
     * Cleanup
     */
    if(dsigCtx != NULL) { 
	xmlDSigCtxDestroy(dsigCtx);
    }
    if(doc != NULL) {
	xmlFreeDoc(doc); 
    }
    
    if(keyMgr != NULL) {
	xmlSecKeyMngrDestroy(keyMgr);
    }
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    
    /* 
     * Shutdown OpenSSL
     */
    RAND_cleanup();
    ERR_clear_error();

    return((ret >= 0) ? 0 : 1);
}

