/** 
 * XML Security examples
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/err.h>
#include <openssl/rand.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <xmlsec/keysmgmt.h> 
#include <xmlsec/algorithms.h>
#include <xmlsec/xmldsig.h>



int main(int argc, char **argv) {
    xmlDocPtr doc = NULL;
    xmlDSigCtxPtr dsigCtx = NULL;
    int rnd_seed = 0;
        
    if(argc < 1) {
	fprintf(stderr, "Error: missed required parameter. Usage: %s <xml-file>\n", argv[0]);
	return(1);
    }
    
    /** 
     * Init OpenSSL
     */    
    while (RAND_status() != 1) {
	RAND_seed(&rnd_seed, sizeof(rnd_seed));
    }
    
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION

    /*
     * build an XML tree from a the file; we need to add default
     * attributes and resolve all character and entities references
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    /** 
     * Load doc 
     */
    doc = xmlParseFile(argv[1]);
    if (doc == NULL) {
	fprintf(stderr, "Error	: unable to parse file \"%s\"\n", argv[1]);
	goto done;
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", argv[1]);
	goto done;
    }

    /**
     * Create Signature Context 
     */
    dsigCtx = xmlDSigCtxCreate(NULL, NULL);
    if(dsigCtx == NULL) {
    	fprintf(stderr,"Error: failed to create dsig context\n");
	goto done; 
    }

    /**
     * Verify It!
     */ 
    if(xmlDSigValidate(dsigCtx, doc, NULL, NULL) < 0) {
    	fprintf(stderr,"validation failed\n");
    } else {
    	fprintf(stderr,"validation succeeded\n");
    }
    
done:
    /*
     * Cleanup
     */
    if(dsigCtx != NULL) { 
	xmlDSigCtxDestroy(dsigCtx);
    }
    if(doc != NULL) {
	xmlFreeDoc(doc); 
    }
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    
    /* 
     * Shutdown OpenSSL
     */
    RAND_cleanup();
    ERR_clear_error();

    return(0);
}

