/** 
 * XMLSec library
 *
 * Binary transforms and binary/xml transforms chain
 *
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#ifndef __XMLSEC_TRANSFORMS_H__
#define __XMLSEC_TRANSFORMS_H__    

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */ 

#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/parser.h>

#include <xmlsec/algorithms.h>
#include <xmlsec/c14n.h>


#define XMLSEC_DIRECTION_READ	1
#define XMLSEC_DIRECTION_WRITE	2

/**
 *
 * Binary transforms are very similar to BIO from OpenSSL.
 * However,there is one big difference. In OpenSSL BIO writing 
 * operation *always* encrypts data and read operation *always*
 * decrypts data. We do not want to have this restriction.
 * The encrypt/decrypt is specified as a flag and transform can
 * do encrypt on read or decrypt on write.
 *
 */ 
typedef struct _xmlSecBinTransform xmlSecBinTransform, *xmlSecBinTransformPtr;

typedef void (*xmlSecBinTransformDestroyCallback)(xmlSecBinTransformPtr self);
typedef int  (*xmlSecBinTransformReadCallback)(xmlSecBinTransformPtr self, 
					        unsigned char *buf, 
					        size_t size);
typedef int  (*xmlSecBinTransformWriteCallback)(xmlSecBinTransformPtr self, 
					        const unsigned char *buf, 
						size_t size);
typedef int  (*xmlSecBinTransformFlushCallback)(xmlSecBinTransformPtr self);

struct _xmlSecBinTransform {
    /* what are we doing */
    xmlSecAlgorithm			algorithm;
    int					encode;
    int					finalized;
    
    /* our position in transforms chain */
    xmlSecBinTransformPtr		prev;
    xmlSecBinTransformPtr		next;
    
    /* functions */
    xmlSecBinTransformDestroyCallback	destroyCallback;
    xmlSecBinTransformReadCallback	readCallback;
    xmlSecBinTransformWriteCallback	writeCallback;
    xmlSecBinTransformFlushCallback	flushCallback;
    
    /* transform specific data */
    void				*data;
};

xmlSecBinTransformPtr	xmlSecBinTransformAdd		(xmlSecBinTransformPtr cur,
							 xmlSecBinTransformPtr ptr);
xmlSecBinTransformPtr	xmlSecBinTransformAddBefore	(xmlSecBinTransformPtr cur,
		    					 xmlSecBinTransformPtr ptr);
void			xmlSecBinTransformRemove	(xmlSecBinTransformPtr ptr);
void			xmlSecBinTransformDestroy	(xmlSecBinTransformPtr ptr);
void			xmlSecBinTransformDestroyAll	(xmlSecBinTransformPtr ptr);
int 			xmlSecBinTransformRead		(xmlSecBinTransformPtr self, 
							 unsigned char *buf, 
							 size_t size);
int 			xmlSecBinTransformWrite		(xmlSecBinTransformPtr self, 
							 const unsigned char *buf, 
							 size_t size);
int 			xmlSecBinTransformFlush		(xmlSecBinTransformPtr self);

xmlOutputBufferPtr xmlSecBinTransformCreateOutputBuffer
							(xmlSecBinTransformPtr ptr,
							 xmlCharEncodingHandlerPtr encoder);

/**
 * Binary/XML transforms chain
 */

typedef struct _xmlSecTransform {
    xmlDocPtr			initDoc;
    xmlNodeSetPtr		initNodeSet;
    const char			*initUri;
    
    xmlSecBinTransformPtr	first;
    xmlSecBinTransformPtr	last;
    
    /* xml doc */
    xmlDocPtr			curDoc;
    xmlNodeSetPtr		curNodeSet;
    
    /* optimization: special case for c14n algorithms */
    xmlSecC14NTransformPtr	c14n;
    
    /* bin stream */
    xmlBufferPtr		curBuf;
} xmlSecTransform, *xmlSecTransformPtr;

xmlSecTransformPtr	xmlSecTransformCreate		(xmlDocPtr doc,
							 xmlNodeSetPtr nodeSet,
							 const char *uri);
void			xmlSecTransformDestroy		(xmlSecTransformPtr ctx);							 
int			xmlSecTransformUpdate		(xmlSecTransformPtr ctx,
							 xmlSecAlgorithm alg,
							 int encode,
							 xmlNodePtr transformNode,
							 void *key,
							 void *data);
int 			xmlSecTransformFinal		(xmlSecTransformPtr ctx,
							 int binary);
							 
#ifdef __cplusplus	
}
#endif /* __cplusplus */

#endif /* __XMLSEC_TRANSFORMS_H__ */

