/** 
 * XMLSec library
 *
 * Supported Algorithms definitions
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <xmlsec/algorithms.h>

/*
 * The list of supported algorithms with properties 
 *
 * http://www.w3.org/TR/xmldsig-core/#sec-AlgID:
 * http://www.w3.org/TR/xml-exc-c14n#sec-Use:
 */
typedef struct _xmlSecStringAlgorithm {
    xmlSecAlgorithm	method;
    xmlSecAlgUsage	usage;
    const xmlChar	*shortName;
    const xmlChar	*href;
}xmlSecStringAlgorithm;

static const xmlSecStringAlgorithm algorithms[] = {

    /* digest methods */ 
    { xmlSecDigestSha1, xmlSecAlgUsageDSigDigest,
	BAD_CAST "SHA1", BAD_CAST "http://www.w3.org/2000/09/xmldsig#sha1" },

    /* encoding */
    { xmlSecEncBase64, xmlSecAlgUsageDSigTransform,
	BAD_CAST "BASE64", BAD_CAST "http://www.w3.org/2000/09/xmldsig#base64" },

    /* MAC */
    { xmlSecMacHmacSha1, xmlSecAlgUsageDSigSignature,
	BAD_CAST "HMAC-SHA1", BAD_CAST "http://www.w3.org/2000/09/xmldsig#hmac-sha1" },
        
    /* signature */
    { xmlSecSignDsaSha1, xmlSecAlgUsageDSigSignature,
	BAD_CAST "DSA-SHA1", BAD_CAST "http://www.w3.org/2000/09/xmldsig#dsa-sha1" },
    { xmlSecSignRsaSha1, xmlSecAlgUsageDSigSignature,
	BAD_CAST "RSA-SHA1", BAD_CAST "http://www.w3.org/2000/09/xmldsig#rsa-sha1" },
    
    /* c14n methods */
    { xmlSecC14NInclusive, xmlSecAlgUsageDSigC14N | xmlSecAlgUsageDSigTransform,
	BAD_CAST "C14N", BAD_CAST "http://www.w3.org/TR/2001/REC-xml-c14n-20010315" },
    { xmlSecC14NInclusiveWithComments, xmlSecAlgUsageDSigC14N | xmlSecAlgUsageDSigTransform,
	BAD_CAST "C14N-with-comments", BAD_CAST "http://www.w3.org/TR/2001/REC-xml-c14n-20010315#WithComments" },
    { xmlSecC14NExclusive, xmlSecAlgUsageDSigC14N | xmlSecAlgUsageDSigTransform,
	BAD_CAST "Exc-C14N", BAD_CAST "http://www.w3.org/2001/10/xml-exc-c14n#" },
    { xmlSecC14NExclusiveWithComments, xmlSecAlgUsageDSigC14N | xmlSecAlgUsageDSigTransform,
	BAD_CAST "Exc-C14N-with-comments", BAD_CAST "http://www.w3.org/2001/10/xml-exc-c14n#WithComments" },

    /* transforms */
    { xmlSecTransformXslt, xmlSecAlgUsageDSigTransform,
	BAD_CAST "XSLT", BAD_CAST "http://www.w3.org/TR/1999/REC-xslt-19991116 " },
    { xmlSecTransformXpath, xmlSecAlgUsageDSigTransform,
	BAD_CAST "XPath", BAD_CAST "http://www.w3.org/TR/1999/REC-xpath-19991116" },
    { xmlSecTransformEnvSign, xmlSecAlgUsageDSigTransform,
	BAD_CAST "EnvSign", BAD_CAST "http://www.w3.org/2000/09/xmldsig#enveloped-signature" },
    
    /* encryption algorithms */

    /* invalid/unknown */
    { xmlSecAlgorithmInvalid, xmlSecAlgUsageUnknown, 
	BAD_CAST "Invalid", NULL },
    { xmlSecAlgorithmUnknown, xmlSecAlgUsageUnknown, 
	BAD_CAST "Unknown", NULL }
};

/**
 * xmlSecHref2Algorithm:
 * @href:	the algorithm href
 *
 * Searches the algorithms tables for an algorithm with 
 * given href.
 *
 * Returns the algorithm with specified href or xmlSecAlgorithmUnknown
 */
xmlSecAlgorithm
xmlSecHref2Algorithm(const xmlChar* href) {
    int i;
    
    if(href == NULL) {
	return(xmlSecAlgorithmInvalid);	
    }
    
    for(i = 0; algorithms[i].href != NULL; ++i) {
	if(strcmp(algorithms[i].href, href) == 0) {
	    return(algorithms[i].method);
	}
    }
    return(xmlSecAlgorithmUnknown);
}

/**
 * xmlSecAlgorithm2Href:
 * @method:	the algorithm
 *
 * Searches the algorithms tables for an algorithm
 *
 * Returns the href for given algorithm
 */
const xmlChar* 
xmlSecAlgorithm2Href(xmlSecAlgorithm method) {
    int i;
    
    for(i = 0; algorithms[i].href != NULL; ++i) {
	if(algorithms[i].method == method) {
	    return(algorithms[i].href);
	}
    }
    return(NULL);
}

/**
 * xmlSecAlgCheckUsage:
 * @method:	the algorithm
 * @usage: 	the set of usage flags produced with "|" operation
 *
 * Checks whether the algorithm could be used as specified
 *
 * Returns 1 if the usage is allowed for algorithm and 0 otherwise.
 */
int
xmlSecAlgCheckUsage(xmlSecAlgorithm method, xmlSecAlgUsage usage) {
    int i;
    
    for(i = 0; algorithms[i].href != NULL; ++i) {
	if(algorithms[i].method == method) {
	    return((algorithms[i].usage & usage) == usage);
	}
    }
    return(0);
}

/**
 * xmlSecAlgCheckUsageHref:
 * @href: 	the algorithms href
 * @method:	the pointer to method storage 
 * usage:	the set of usage flags produced with "|" operation 
 *
 * Searches the supported algorithms for the algorithm with given 
 * href and checks that it could be used as specified.
 * 
 * Returns 1 if the usage is allowed for algorithm and 0 otherwise.
 */

int
xmlSecAlgCheckUsageHref(const xmlChar* href, xmlSecAlgorithm* method,
		       xmlSecAlgUsage usage) {
    int i;
    
    if(href == NULL) {
	return(xmlSecAlgorithmInvalid);	
    }

    if(method) (*method) = xmlSecAlgorithmUnknown;    
    for(i = 0; algorithms[i].href != NULL; ++i) {
	if(strcmp(algorithms[i].href, href) == 0) {
	    if(((algorithms[i].usage) & usage) == usage) {
		if(method != NULL)  {
		    (*method) = algorithms[i].method;
		}
		return(1);
	    }
	    return(0);
	}
    }
    return(-1);
}

/**
 * xmlSecAlgCheckUsageAlg:
 * @method:	the algorithm
 * @usage:	the set of usage flags produced with "|" operation  
 *
 * Searches the supported algorithms for the specified algorithms,
 * checks that it could be used as specified and returns the href 
 * for specified algorithm
 *
 * Returns the href for specified algorithm or NULL if it is not found or
 * could not be used as specified.
 */
const xmlChar* 		
xmlSecAlgCheckUsageAlg(xmlSecAlgorithm method, xmlSecAlgUsage usage) {
    int i;
    
    for(i = 0; algorithms[i].href != NULL; ++i) {
	if(algorithms[i].method == method) {
	    if(((algorithms[i].usage) & usage) == usage) {
		return(algorithms[i].href);
	    }
	    return(NULL);
	}
    }
    return(NULL);
}

