/** 
 * Base64 Algorithm
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/xmlerror.h> 

#include <xmlsec/base64.h>


/* 
 * the table to map numbers to base64 
 */
static const unsigned char base64[] =
{  
/*   0    1    2    3    4    5    6    7   */
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', /* 0 */
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', /* 1 */
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', /* 2 */
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', /* 3 */
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', /* 4 */
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v', /* 5 */
    'w', 'x', 'y', 'z', '0', '1', '2', '3', /* 6 */
    '4', '5', '6', '7', '8', '9', '+', '/'  /* 7 */
};


/* few macros to simplify the code */
#define	XMLSEC_BASE64_BUFFER_SIZE	8

#define xmlSecBase64Encode1(a) 		(base64[((a) >> 2) & 0x3F])
#define xmlSecBase64Encode2(a, b) 	(base64[(((a) << 4) & 0x30) + (((b) >> 4) & 0x0F)])
#define xmlSecBase64Encode3(b, c) 	(base64[(((b) << 2) & 0x3c) + (((c) >> 6) & 0x03)])
#define xmlSecBase64Encode4( c)		(base64[(c) & 0x3F])

#define xmlSecBase64Decode1(a, b)	(((a) << 2) | (((b) & 0x3F) >> 4))
#define xmlSecBase64Decode2(b, c)	(((b) << 4) | (((c) & 0x3F) >> 2))
#define xmlSecBase64Decode3(c, d)	(((c) << 6) | ((d) & 0x3F))
	
#define xmlSecIsBase64Char(ch) 		((((ch) >= 'A') && ((ch) <= 'Z')) || \
					 (((ch) >= 'a') && ((ch) <= 'z')) || \
					 (((ch) >= '0') && ((ch) <= '9')) || \
					  ((ch) == '+') || ((ch) == '/')) 

/**
 *
 * Base64 Context
 *
 */

struct _xmlSecBase64Ctx {
    int			encode;
    
    unsigned char	in[4];
    unsigned char	out[16];
    size_t 		inPos;
    
    size_t		linePos;
    size_t		columns;    
    int			equalSigns;
};

/**
 * xmlSecBase64CtxEncode:
 * @ctx:	the base64 context
 *
 * Encodes data stored in the context
 *
 * Returns 0 for success and -1 otherwise.
 */
static int
xmlSecBase64CtxEncode(xmlSecBase64CtxPtr ctx) {
    int outPos;
    
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxEncode: context is null\n");	
#endif
	return(-1);
    }
    if(ctx->inPos == 0) {
	return(0); /* nothing to encode */
    }

    outPos = 0;    
    if(ctx->columns > 0 && ctx->columns <= ctx->linePos) {
	ctx->out[outPos++] = '\n';
	ctx->linePos = 0;
    }
    ++ctx->linePos;
    ctx->out[outPos++] = xmlSecBase64Encode1(ctx->in[0]);

    if(ctx->columns > 0 && ctx->columns <= ctx->linePos) {
	ctx->out[outPos++] = '\n';
	ctx->linePos = 0;
    }
    ++ctx->linePos;
    ctx->out[outPos++] = xmlSecBase64Encode2(ctx->in[0], ctx->in[1]);

    if(ctx->columns > 0 && ctx->columns <= ctx->linePos) {
	ctx->out[outPos++] = '\n';
	ctx->linePos = 0;
    }
    ++ctx->linePos;
    ctx->out[outPos++] = (ctx->inPos > 1) ? xmlSecBase64Encode3(ctx->in[1], ctx->in[2]) : '=';

    if(ctx->columns > 0 && ctx->columns <= ctx->linePos) {
	ctx->out[outPos++] = '\n';
	ctx->linePos = 0;
    }
    ++ctx->linePos;
    ctx->out[outPos++] = (ctx->inPos > 2) ? xmlSecBase64Encode4(ctx->in[2]) : '=';
    	    
    ctx->inPos = 0;    
    return(outPos);
}

/**
 * xmlSecBase64CtxDecode:
 * @ctx:	the base64 context
 *
 * Decodes data stored in the context
 *
 * Returns 0 for success and -1 otherwise.
 */
static int
xmlSecBase64CtxDecode(xmlSecBase64CtxPtr ctx) {
    int outPos;
    
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxDecode: context is null\n");	
#endif
	return(-1);
    }
    
    outPos = 0;
    if(ctx->inPos == 0) {
	return(0); /* nothing to decode */
    }
    if(ctx->inPos < 2) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxDecode: invalid format (only one or two equal signs are allowed at the end)\n");	
	return(-1);
    }
    ctx->out[outPos++] = xmlSecBase64Decode1(ctx->in[0], ctx->in[1]);

    if(ctx->inPos > 2) {
	ctx->out[outPos++] = xmlSecBase64Decode2(ctx->in[1], ctx->in[2]);
	if(ctx->inPos > 3) {
	    ctx->out[outPos++] = xmlSecBase64Decode3(ctx->in[2], ctx->in[3]);
	}
    }
    ctx->inPos = 0;
    return(outPos);
}

/**
 * xmlSecBase64CtxCreate:
 * @encode:	the encode/decode flag (1 - encode, 0 - decode) 
 * @columns:	the number of columns (lines length) in the output
 *		if this parameter is 0 then no line breaks inserted
 *
 * Creates base64 context.
 *
 * Returns a pointer to newly created base64 context (caller is responsible
 * for deleting it using xmlSecBase64CtxDestroy) or NULL if an error occurs.
 */
xmlSecBase64CtxPtr	
xmlSecBase64CtxCreate(int encode, int columns) {
    xmlSecBase64CtxPtr ctx;
    
    /*
     * Allocate a new xmlSecBase64CtxPtr and fill the fields.
     */
    ctx = (xmlSecBase64CtxPtr) xmlMalloc(sizeof(xmlSecBase64Ctx));
    if (ctx == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxCreate: malloc failed\n");
	return(NULL);
    }
    memset(ctx, 0, sizeof(xmlSecBase64Ctx));

    ctx->encode = encode;
    ctx->columns = columns;
    return(ctx);
}

/**
 * xmlSecBase64CtxDestroy:
 * @ctx:	the base64 context
 * 
 * Destroys base64 context.
 */
void
xmlSecBase64CtxDestroy(xmlSecBase64CtxPtr ctx) {
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxDestroy: context is null\n");	
#endif
	return;
    }
    
     memset(ctx, 0, sizeof(xmlSecBase64Ctx)); 
     xmlFree(ctx);
}

/**
 * xmlSecBase64CtxInit:
 * @ctx:	the base64 context
 *
 * Resets base64 context. After calling this function the 
 * context could be re-used for new encoding/decoding
 *
 * Returns 0 for success and -1 otherwise
 */
int
xmlSecBase64CtxInit(xmlSecBase64CtxPtr ctx) {
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxInit: context is null\n");	
#endif
	return(-1);
    }

    memset(ctx->in, 0, sizeof(ctx->in));
    memset(ctx->out, 0, sizeof(ctx->out));
    ctx->inPos = ctx->linePos = 0;
    ctx->equalSigns = 0;   
    return(0);
}

/**
 * xmlSecBase64CtxUpdate:
 * @ctx:	the base64 context
 * @in:		the input buffer
 * @inLen:	the input buffer size
 * @out:	the output buffer
 * @outLen:	the output buffer size
 *
 * Encodes/decodes the next piece of data from input buffer.
 * 
 * Returns the number of bytes written to output buffer or 
 * -1 if an error occurs.
 */
int
xmlSecBase64CtxUpdate(xmlSecBase64CtxPtr ctx,
		     const unsigned char *in, size_t inLen, 
		     unsigned char *out, size_t outLen) {
    unsigned char ch;
    size_t inPos, outPos;
    size_t size;
    int ret;
    
    if((ctx == NULL) || (out == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxUpdate: context or out is null\n");	
#endif
	return(-1);
    }
    
    if((in == NULL) || (inLen == 0)) {
	return(0);
    }    
        
    inPos = outPos = 0;
    size = (ctx->encode) ? 3 : 4;
    
    /* if we have something in in process this first */
    while(inPos < inLen) {
	if(ctx->inPos >= size) {
	    /* do encode/decode */
	    if(ctx->encode) {
		ret = xmlSecBase64CtxEncode(ctx);
	    } else {
		ret = xmlSecBase64CtxDecode(ctx);
	    }
	    if(ret < 0) {
#ifdef DEBUG_XMLSEC
		xmlGenericError(xmlGenericErrorContext,
		    "xmlSecBase64CtxUpdate: encode/decode failed\n"); 
#endif		    
		return(-1);
	    }
	    
	    if(outPos + ret > outLen) {
		xmlGenericError(xmlGenericErrorContext,
		    "xmlSecBase64CtxUpdate: buffer is too small (%d > %d)\n", outPos + ret, outLen); 
		return(-1);
	    }	    
	    memcpy(out + outPos, ctx->out, ret);
	    outPos += ret;
	}
	
	/* read next char in the buffer */
	ch = in[inPos++];
	if(ctx->encode) {
	    ctx->in[ctx->inPos++] = ch;
	} else if(xmlSecIsBase64Char(ch)) {
	    if(ctx->equalSigns != 0) {
		xmlGenericError(xmlGenericErrorContext, 
		    "xmlSecBase64CtxUpdate: only space characters are allowed after equal sign \'=\'\n");
		return(-1);    
	    }
	    if((ch >= 'A') && (ch <= 'Z')) {
		ctx->in[ctx->inPos++] = (ch - 'A');
	    } else if((ch >= 'a') && (ch <= 'z')) {
		ctx->in[ctx->inPos++] = 26 + (ch - 'a');
	    } else if((ch >= '0') && (ch <= '9')) {
		ctx->in[ctx->inPos++] = 52 + (ch - '0'); 
	    } else if(ch == '+') {
		ctx->in[ctx->inPos++] = 62;
	    } else if(ch == '/') {
		ctx->in[ctx->inPos++] = 63;
	    }	    
	} else if(ch == '='){
	    if(ctx->equalSigns >= 2) {
		xmlGenericError(xmlGenericErrorContext, 
		    "xmlSecBase64CtxUpdate: too many equal signs at the end (most of two accepted)\n");
		return(-1);    
	    }
	    ++ctx->equalSigns;
	}
    }
        
    return(outPos);
}

/**
 * xmlSecBase64CtxFinal:
 * @ctx:	the base64 context
 * @out:	the output buffer
 * @outLen:	the output buffer size
 *
 * Encodes/decodes the last piece of data stored in the context
 * and finalizes the result.
 *
 * Returns the number of bytes written to output buffer or 
 * -1 if an error occurs.
 */
int
xmlSecBase64CtxFinal(xmlSecBase64CtxPtr ctx, 
		    unsigned char *out, size_t outLen) {
    int ret;
    size_t size;
            
    if((ctx == NULL) || (out == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxFinal: context or out is null\n");	
#endif
	return(-1);
    }

    /* zero uninitialized input bytes */
    size = (ctx->encode) ? 3 : 4;
    memset(ctx->in + ctx->inPos, 0, size - ctx->inPos);

    /* do encode/decode */
    if(ctx->encode) {
	ret = xmlSecBase64CtxEncode(ctx);
    } else {
	ret = xmlSecBase64CtxDecode(ctx);
    }
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxFinal: encode/decode failed\n"); 
#endif		    
	return(-1);
    }
	    
    /* copy to out put buffer */
    if((size_t)ret > outLen) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64CtxUpdate: buffer is too small (%d > %d)\n", ret, outLen); 
	return(-1);
    }	    
    if(ret > 0) {
	memcpy(out, ctx->out, ret);
    }    
#if 0
    /* add \n at the end of decoding (todo: do we need it?) */
    if(ctx->encode && (ctx->columns > 0) && ((ret + 1) < outLen)) {
	out[ret++] = '\n';
    }
#endif    
    /* add \0 */
    if((size_t)(ret + 1) < outLen) {
	out[ret] = '\0';
    }
    return(ret);
}


/**
 *
 * Base64 binary transform
 *
 */
 
/**
 * xmlSecBase64TransformRead:
 * @ptr: 	the base64 transform
 * @buf:	the pointer to output buffer
 * @len:	the output buffer size
 *
 * Reads data from previous tranform, encodes/decodes them
 * and returns the result in the output buffer
 * 
 * Returns the number of bytes written to output buffer or 
 * -1 if an error occurs. 
 */
static int
xmlSecBase64TransformRead(xmlSecBinTransformPtr ptr, 
			  unsigned char *buf, size_t len) {
    int ret;
    int size;
    xmlSecBase64CtxPtr ctx;        
    
    if((ptr == NULL) || (buf == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformRead: ptr, ptr->data or buffer is null\n");
#endif	    
	return(-1);
    }
    
    ctx = (xmlSecBase64CtxPtr)ptr->data;

    /* if we already called Final then nothing to read more! */
    if(ptr->finalized) {
	return(0);
    }
    
    /* estimate the input buffer size to produce len bytes */    
    if(ptr->encode) {
	/* the encoded data are 4/3 input + line breaks */
	if(ctx->columns > 0) {
	    size = 3 * (len - (len / ctx->columns) - 1) / 4 - 8;
	} else {
	    size = (3 * len) / 4 - 8;
	}
    } else {
	/* the decode data len is always less than input */
	size = len - 8;
    }
    if(size < 0) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformRead: buffer size %d is too small\n", len);
	return(-1);
    }
    
    ret = xmlSecBinTransformRead(ptr->prev, buf + len - size, size);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformRead: prev read failed\n");
#endif 	    
	return(-1);
    }
    
    if(ret > 0) {
	ret = xmlSecBase64CtxUpdate(ctx, buf + len - size, ret, buf, len);
    } else { 
	/* ret == 0 i.e. there is no more data */
	ret = xmlSecBase64CtxFinal(ctx, buf, len);
	ptr->finalized = 1;
    } 
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformRead: update or final failed\n");
#endif 	    
	return(-1);
    }
    
    return(ret);
}

/**
 * xmlSecBase64TransformWrite:
 * @ptr:	the base64 transform
 * @buf:	the input buffer
 * @len:	the input buffer size
 *
 * Encodes/decodes data from input buffer and writes result to the next
 * transform in chain,
 *
 * Returns the number of bytes written to the next transform or 
 * -1 if an error occurs.  
 */
static int
xmlSecBase64TransformWrite(xmlSecBinTransformPtr ptr, 
			   const unsigned char *buf, size_t len) {
    xmlSecBase64CtxPtr ctx;  
    int ret = 0;
    size_t size;
    unsigned char buffer[XMLSEC_BASE64_BUFFER_SIZE*6];
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }
    if(buf == NULL) {
	return(0);
    }
    
    ctx = (xmlSecBase64CtxPtr)ptr->data;
    
    while(len > 0) {
	size = (len > XMLSEC_BASE64_BUFFER_SIZE) ? XMLSEC_BASE64_BUFFER_SIZE : len;
	ret = xmlSecBase64CtxUpdate(ctx, buf, size, buffer, sizeof(buffer));
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecBase64TransformWrite: update failed\n");
#endif 	    
	    return(-1);
	}
	
	ret = xmlSecBinTransformWrite(ptr->next, buffer, ret);
        if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecBase64TransformWrite: prev write failed\n");
#endif 	    
	    return(-1);
	}	 
	buf += size;
	len -= size;
    }
    
    return(ret);
}

/**
 * xmlSecBase64TransformFlush:
 * @ptr:	the base64 transform
 *
 * Encodes/decodes the last piece of data stored in the context, 
 * writes result to next buffer and calls flush for it.
 *
 * Returns the number of bytes written to the next buffer or 
 * -1 if an error occurs
 */
static int
xmlSecBase64TransformFlush(xmlSecBinTransformPtr ptr) {
    xmlSecBase64CtxPtr ctx;  
    unsigned char buffer[XMLSEC_BASE64_BUFFER_SIZE*6];
    int ret = 0;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformFlush: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    ctx = (xmlSecBase64CtxPtr)ptr->data;
    
    ret = xmlSecBase64CtxFinal(ctx, buffer, sizeof(buffer));
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformFlush: final failed\n");
#endif 	    
	return(-1);
    }
	
    ret = xmlSecBinTransformWrite(ptr->next, buffer, ret);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecBase64TransformFlush: prev write failed\n");
#endif 	    
	return(-1);
    }	 

    ret = xmlSecBinTransformFlush(ptr->next);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformFlush: prev flush failed\n");
#endif 	    
	return(-1);
    } 
    return(0);
}

 
/**
 * xmlSecBase64TransformDestroy:
 * @ptr:	the base64 transform
 *
 * Destroys and base64 transform and frees the used memory
 *
 */
static void 
xmlSecBase64TransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    if(ptr->data != NULL) {
	xmlSecBase64CtxDestroy((xmlSecBase64CtxPtr)ptr->data);
    }    
    xmlFree(ptr);
}


/**
 * xmlSecBase64TransformCreate:
 * @transformNode:	the base64 transform node
 * @encode:	the encode (1) / decode (0) flag
 *
 * Creates new base64 transform and reads the parameters from
 * the given node.
 *
 * Returns the newly created base64 transform (caller is responsible
 * for deleting it using xmlSecTransformDestroy function) or NULL 
 * if an error occurs.
 */
xmlSecBinTransformPtr	
xmlSecBase64TransformCreate(xmlNodePtr transformNode, int encode) {
    xmlSecBinTransformPtr ptr;
    int columns = XMLSEC_BASE64_LINESIZE;
    /*
     * Allocate a new xmlSecBase64Transform and fill the fields.
     */
    ptr = (xmlSecBinTransformPtr) xmlMalloc(sizeof(xmlSecBinTransform));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform));

    if(transformNode != NULL) {
	/* todo: read the number of columns */
    }
    
    ptr->data = xmlSecBase64CtxCreate(encode, columns);
    if(ptr->data == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64TransformCreate: xmlSecBase64CtxCreate failed\n");
#endif 	    
	xmlSecBase64TransformDestroy(ptr);	
	return(NULL);
    }
    
    ptr->algorithm = xmlSecEncBase64;
    ptr->encode = encode;
    
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecBase64TransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecBase64TransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecBase64TransformWrite;
    ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecBase64TransformFlush;
    
    return(ptr);    
}

/**
 * xmlSecBase64Encode:
 * @buf:	the input buffer
 * @len:	the input buffer size
 * @columns:	the output max line length (if 0 then no line breaks
 *		would be inserted)
 *
 * Encodes the data from input buffer and allocates the string for the result
 *
 * Returns newly allocated string with base64 encoded data (caller is 
 * responsibe for freeing it) or NULL if an error occurs.
 */
xmlChar*
xmlSecBase64Encode(const unsigned char *buf, size_t len, int columns) {
    xmlSecBase64CtxPtr ctx;
    xmlChar *ptr;
    size_t size;    
    int size_update, size_final;
    int ret;

    if(buf == NULL) {	
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Encode: buf is NULL\n");
#endif 	    
	return(NULL);	
    }    
    ctx = xmlSecBase64CtxCreate(1, columns);
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Encode: failed to create context NULL\n");
#endif 	    
	return(NULL);
    }
    
    /* create result buffer */
    size = (4 * len) / 3 + 4;
    if(columns > 0) {
	size += (size / columns) + 4;
    }
    ptr = (xmlChar*) xmlMalloc(size);
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Encode: failed to allocate memory for result\n");
#endif 	    
	xmlSecBase64CtxDestroy(ctx);
	return(NULL);
    }

    ret = xmlSecBase64CtxUpdate(ctx, buf, len, (unsigned char*)ptr, size);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Encode: base64 update failed\n");
#endif 	    
	xmlFree(ptr);
	xmlSecBase64CtxDestroy(ctx);
	return(NULL);
    }
    size_update = ret;

    ret = xmlSecBase64CtxFinal(ctx, ((unsigned char*)ptr) + size_update, size - size_update);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Encode: base64 final failed\n");
#endif 	    
	xmlFree(ptr);
	xmlSecBase64CtxDestroy(ctx);
	return(NULL);
    }
    size_final = ret;
    ptr[size_update + size_final] = '\0';
    
    xmlSecBase64CtxDestroy(ctx);
    return(ptr);
}

/**
 * xmlSecBase64Decode:
 * @str:	the input buffer with base64 encoded string
 * @buf:	the output buffer
 * @len:	the output buffer size
 *
 * Decodes input base64 encoded string and puts result into
 * the output buffer.
 *
 * Returns the number of bytes written to the output buffer or 
 * -1 if an error occurs 
 */
int
xmlSecBase64Decode(const xmlChar* str, unsigned char *buf, size_t len) {
    xmlSecBase64CtxPtr ctx;
    int size_update;
    int size_final;
    int ret;

    if((str == NULL) || (buf == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Decode: str or buf is NULL\n");
#endif 	    
	return(-1);	
    }
    
    ctx = xmlSecBase64CtxCreate(0, 0);
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Decode: failed to create context NULL\n");
#endif 	    
	return(-1);
    }
    
    ret = xmlSecBase64CtxUpdate(ctx, (const unsigned char*)str, xmlStrlen(str), buf, len);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Decode: base64 update failed\n");
#endif 	    
	xmlSecBase64CtxDestroy(ctx);
	return(-1);
    }

    size_update = ret;
    ret = xmlSecBase64CtxFinal(ctx, buf + size_update, len - size_update);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBase64Decode: base64 final failed\n");
#endif 	    
	xmlSecBase64CtxDestroy(ctx);
	return(-1);
    }
    size_final = ret;    

    xmlSecBase64CtxDestroy(ctx);
    return(size_update + size_final);
}


