/** 
 * XMLSec library
 *
 * C14N transforms
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/globals.h>
#include <libxml/xmlerror.h> 
#include <libxml/c14n.h> 

#include <xmlsec/xmlsec.h>
#include <xmlsec/c14n.h>


static const xmlChar xmlExcC14NNs[] = "http://www.w3.org/2001/10/xml-exc-c14n#";
static const xmlChar xmlExcC14NWithCommentsNs[] = "http://www.w3.org/2001/10/xml-exc-c14n#WithComments";

/**
 * XML C14N transform
 */
struct _xmlSecC14NTransform {
    int			exclusive;
    int			withComments;

    /* exc transforms */
    xmlChar		*inclusiveNsBuffer;
    xmlChar		**inclusiveNsList;
};

/**
 * xmlSecC14NTransformInclusivePrefixesListRead:
 * @ptr:	the c41n transform
 * @cur:	the "Trasnform" node
 * 
 * Reads "InclusiveNamespaces" list for exclucive c14n transform from
 * xml node (http://www.w3.org/TR/xml-exc-c14n#sec-Use)
 *
 * Return 0 on success and -1 otherwise.
 */
static int
xmlSecC14NTransformInclusivePrefixesListRead(xmlSecC14NTransformPtr ptr, xmlNodePtr cur) {
    xmlChar *p;
    size_t count, len;
    
    if((ptr == NULL) || (cur == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformInclusivePrefixesListRead: ptr or cur is null\n");
#endif 	    
	return(-1);
    }
    /* find InclusiveNamespaces children */
    cur = xmlSecGetNextElementNode(cur->children);
    while(cur != NULL){ 
	if(xmlSecCheckNodeName(cur->doc, cur, BAD_CAST "InclusiveNamespaces", xmlExcC14NNs) ||
	   xmlSecCheckNodeName(cur->doc, cur, BAD_CAST "InclusiveNamespaces", xmlExcC14NWithCommentsNs)) {
		break;
	}
        cur = xmlSecGetNextElementNode(cur->next);
    }
    
    if(cur == NULL) {
	/* nothing found */
	return(0); 
    }
    ptr->inclusiveNsBuffer = xmlGetProp(cur, BAD_CAST "PrefixList");
    if((ptr->inclusiveNsBuffer) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformInclusivePrefixesListRead: InclusiveNamespaces node has no PrefixList attr\n");
#endif 	    
	return(-1);
    }
    
    count = 0;
    len = 0;
    p = ptr->inclusiveNsBuffer;
    while((*p) != '\0') {
	if(((*p) == ' ') && (len > 0)) {
	    len = 0;
	    ++count;
	} else if((*p) != ' ') {
	    ++len;
	}
	++p;
    }
    
    ptr->inclusiveNsList = (xmlChar**)xmlMalloc(sizeof(xmlChar*) * (count + 2));
    if(ptr->inclusiveNsList == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformInclusivePrefixesListRead: malloc failed\n");
#endif 	    
	return(-1);
    }
    memset(ptr->inclusiveNsList, 0, sizeof(xmlChar*) * (count + 2));

    count = 0;
    len = 0;
    p = ptr->inclusiveNsBuffer;
    (ptr->inclusiveNsList)[0] = p;
    while((*p) != '\0') {
	if(((*p) == ' ') && (len > 0)) {
	    (*p) = '\0';
	    len = 0;
	    (ptr->inclusiveNsList)[++count] = p + 1;
	} else if((*p) != ' ') {
	    ++len;
	}
	++p;
    }        
    return(0);
}

/**
 * xmlSecC14NTransformCreate:
 * @alg:	the c14n algorithm method
 * @node:	the "Transform" node for this trasnform
 *
 * Creates new C14N trasnform.
 *
 * Returns the newly create C14N transform (caller is responsible 
 * for freeing the memory using xmlSecC14NTransformDestroy) or 
 * NULL if an error occurs.
 */
xmlSecC14NTransformPtr	
xmlSecC14NTransformCreate(xmlSecAlgorithm alg, xmlNodePtr node) {
    xmlSecC14NTransformPtr ptr;	
    int ret;
    
    /*
     * Allocate a new xmlSecC14NTransform and fill the fields.
     */
    ptr = (xmlSecC14NTransformPtr) xmlMalloc(sizeof(xmlSecC14NTransform));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecC14NTransform));
    
    switch(alg) {
    case xmlSecC14NInclusive:
	ptr->exclusive = 0;
	ptr->withComments = 0;
	break;
    case xmlSecC14NInclusiveWithComments:
	ptr->exclusive = 0;
	ptr->withComments = 1;
	break;
    case xmlSecC14NExclusive:
	ptr->exclusive = 1;
	ptr->withComments = 0;
	break;
    case xmlSecC14NExclusiveWithComments:
	ptr->exclusive = 1;
	ptr->withComments = 1;
	break;
    default:
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformCreate: bad C14N algorithm %d\n", alg);
#endif 	    
	xmlSecC14NTransformDestroy(ptr);
	return(NULL);
    }
    
    if(node != NULL) {
	ret = xmlSecC14NTransformInclusivePrefixesListRead(ptr, node);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecC14NTransformCreate: failed to read inclusive prefixes list\n");
#endif 	    
	    xmlSecC14NTransformDestroy(ptr);
	    return(NULL);
	}
    }
    /* todo: read inclusive prefixes list for exclusive transforms */
    return(ptr);
}

/**
 * xmlSecC14NTransformDestroy:
 * @ptr:	the C14N transform
 *
 * Destroys the C14N transform and frees the allocated memory.
 */
void
xmlSecC14NTransformDestroy(xmlSecC14NTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformDestroy: ptr is null\n");
#endif 	    
	return;
    }
    
    if(ptr->inclusiveNsList != NULL) {
	xmlFree(ptr->inclusiveNsList);
    }
    if(ptr->inclusiveNsBuffer != NULL) {
	xmlFree(ptr->inclusiveNsBuffer);
    }    
    xmlFree(ptr);
}

/**
 * xmlSecC14NTransformExecute:
 * @ptr:	the C14N tranform; if this parameter is NULL then 
 *		default Inclusive C14N w/o comments transform
 *		is executed
 * @doc:	the input document
 * @nodes:	the input document nodes set 
 * @output:	the output buffer for the result
 *
 * Executes the specified or default C14N transform and writes result
 * octet stream to the buffer.
 *
 * Returns 0 for success or -1 if an error occurs
 */
int
xmlSecC14NTransformExecute(xmlSecC14NTransformPtr ptr, xmlDocPtr doc, 
		    xmlNodeSetPtr nodes, xmlOutputBufferPtr output) {
    int ret;

    if(ptr != NULL) {    
        ret = xmlC14NDocSaveTo(doc, nodes, ptr->exclusive, ptr->inclusiveNsList, 
			   ptr->withComments, output);
    } else {
	ret = xmlC14NDocSaveTo(doc, nodes, 0, NULL, 0, output);
    }
    
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecC14NTransformExecute: c14n failed\n");
#endif 	    
	return(-1);
    }
    return(0);
}

