/** 
 * XMLSec library
 *
 * DSA algorithm support
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/sha.h>

#include <xmlsec/xmlsec.h>
#include <xmlsec/base64.h>
#include <xmlsec/bn.h>

#include <xmlsec/dsa.h>

/**
 *
 * The output of the DSA algorithm consists of a pair of integers usually 
 * referred by the pair (r, s). The signature value consists of the base64 
 * encoding of the concatenation of two octet-streams that respectively result 
 * from the octet-encoding of the values r and s in that order. Integer to 
 * octet-stream conversion must be done according to the I2OSP operation 
 * defined in the RFC 2437 [PKCS1] specification with a l parameter equal to 20. 
 *
 */
 
#define XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE		20

typedef struct _xmlSecDsaContext {
    DSA			*dsa;
    SHA_CTX		sha1;		
    unsigned char	digest[2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE]; /* must be the last one! */
} xmlSecDsaSha1Context, *xmlSecDsaSha1ContextPtr;


/**
 * xmlSecDsaKeyDestroy:
 * @ptr:	the DSA key 
 *
 * Destroys the DSA key structure and frees the allocated memory
 */
static void
xmlSecDsaKeyDestroy(xmlSecDsaKeyPtr ptr) {
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyDestroy: ptr is null\n");
#endif 	    
	return;
    }
    if(ptr->name != NULL) {
	xmlFree(ptr->name);
    }
    if(ptr->dsa != NULL) {
	DSA_free(ptr->dsa);
    }
    xmlFree(ptr);
} 

/**
 * xmlSecDsaKeyCreate:
 * @name:	the key name
 *
 * Creates new DSA key
 *
 * Returns the newly allocated DSA key (caller is responsible for 
 * calling xmlSecKeyDestroy to free the memory) or NULL if an error
 * occurs.
 */
xmlSecDsaKeyPtr
xmlSecDsaKeyCreate(const xmlChar* name) {
    xmlSecDsaKeyPtr ptr;

    /*
     * Allocate a new xmlSecDsaKey and fill the fields.
     */
    ptr = (xmlSecDsaKeyPtr) xmlMalloc(sizeof(xmlSecDsaKey));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecDsaKey));
    ptr->algorithm = xmlSecSignDsaSha1;
    ptr->name = (name != NULL) ? xmlStrdup(name) : NULL;
    ptr->size = sizeof(xmlSecDsaKey);
    ptr->destroyCallback = (xmlSecKeyDestroyCallback)xmlSecDsaKeyDestroy;

    /* create new DSA object */
    ptr->dsa = DSA_new();
    if(ptr->dsa == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigDsaKeyCreate: failed to create DSA object\n");
#endif 	    
	xmlSecKeyDestroy((xmlSecKeyPtr)ptr);
	return(NULL);	
    }    
    return(ptr);    
}

/**
 * http://www.w3.org/TR/xmldsig-core/#sec-DSAKeyValue
 *
 * The DSAKeyValue Element
 *
 * DSA keys and the DSA signature algorithm are specified in [DSS]. 
 * DSA public key values can have the following fields:
 *      
 *   * P - a prime modulus meeting the [DSS] requirements 
 *   * Q - an integer in the range 2**159 < Q < 2**160 which is a prime 
 *         divisor of P-1 
 *   * G - an integer with certain properties with respect to P and Q 
 *   * Y - G**X mod P (where X is part of the private key and not made 
 *	   public) 
 *   * J - (P - 1) / Q 
 *   * seed - a DSA prime generation seed 
 *   * pgenCounter - a DSA prime generation counter
 *
 * Parameter J is available for inclusion solely for efficiency as it is 
 * calculatable from P and Q. Parameters seed and pgenCounter are used in the 
 * DSA prime number generation algorithm specified in [DSS]. As such, they are 
 * optional but must either both be present or both be absent. This prime 
 * generation algorithm is designed to provide assurance that a weak prime is 
 * not being used and it yields a P and Q value. Parameters P, Q, and G can be 
 * public and common to a group of users. They might be known from application 
 * context. As such, they are optional but P and Q must either both appear or 
 * both be absent. If all of P, Q, seed, and pgenCounter are present, 
 * implementations are not required to check if they are consistent and are 
 * free to use either P and Q or seed and pgenCounter. All parameters are 
 * encoded as base64 [MIME] values.
 *     
 * Arbitrary-length integers (e.g. "bignums" such as RSA moduli) are 
 * represented in XML as octet strings as defined by the ds:CryptoBinary type.
 *     
 * Schema Definition:
 *     
 * <element name="DSAKeyValue" type="ds:DSAKeyValueType"/> 
 * <complexType name="DSAKeyValueType"> 
 *   <sequence>
 *     <sequence minOccurs="0">
 *        <element name="P" type="ds:CryptoBinary"/> 
 *        <element name="Q" type="ds:CryptoBinary"/>
 *     </sequence>
 *     <element name="G" type="ds:CryptoBinary" minOccurs="0"/> 
 *     <element name="Y" type="ds:CryptoBinary"/> 
 *     <element name="J" type="ds:CryptoBinary" minOccurs="0"/>
 *     <sequence minOccurs="0">
 *       <element name="Seed" type="ds:CryptoBinary"/> 
 *       <element name="PgenCounter" type="ds:CryptoBinary"/> 
 *     </sequence>
 *   </sequence>
 * </complexType>
 *     
 * DTD Definition:
 *     
 *  <!ELEMENT DSAKeyValue ((P, Q)?, G?, Y, J?, (Seed, PgenCounter)?) > 
 *  <!ELEMENT P (#PCDATA) >
 *  <!ELEMENT Q (#PCDATA) >
 *  <!ELEMENT G (#PCDATA) >
 *  <!ELEMENT Y (#PCDATA) >
 *  <!ELEMENT J (#PCDATA) >
 *  <!ELEMENT Seed (#PCDATA) >
 *  <!ELEMENT PgenCounter (#PCDATA) >
 *
 * ============================================================================
 * 
 * To support reading/writing private keys an X element added (before Y).
 * todo: The current implementation does not support Seed and PgenCounter!
 * by this the P, Q and G are *required*!
 **/ 

/**
 * xmlSecDsaKeyPtr:
 * @doc:	the XML document containing the DSA key value
 * @dsaKeyValueNode:	the "DSAKeyValue" node
 * @name:	the key name
 *
 * Reads the DSAKeyValue node and creates new DSA key.
 *
 * Returns the newly allocated DSA key (caller is responsible for 
 * calling xmlSecKeyDestroy to free the memory) or NULL if an error
 * occurs.
 */
xmlSecDsaKeyPtr
xmlSecDsaKeyRead(const xmlDocPtr doc, const xmlNodePtr dsaKeyValueNode, 
		 const xmlChar* name) {
    xmlSecDsaKeyPtr key = NULL;
    xmlNodePtr cur;
    
    if(dsaKeyValueNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: cur is null\n");
#endif 	    
	return(NULL);
    }    
    cur = xmlSecGetNextElementNode(dsaKeyValueNode->children); 

    /* create new object */
    key = xmlSecDsaKeyCreate(name);
    if(key == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to create xmlSecDsaKey \n");
#endif 	    
	return(NULL);	
    }    
    
    /* first is P node. It is REQUIRED because we do not support Seed and PgenCounter*/
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur,  BAD_CAST "P", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: required element \"P\" missed\n");
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    if(xmlSecNodeGetBNValue(cur, &(key->dsa->p)) == NULL) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to convert element \"P\" value\n");
#endif	    
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is Q node. It is REQUIRED because we do not support Seed and PgenCounter*/
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "Q", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: required element \"Q\" missed\n");
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    if(xmlSecNodeGetBNValue(cur, &(key->dsa->q)) == NULL) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to convert element \"Q\" value\n");
#endif	    
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is G node. It is REQUIRED because we do not support Seed and PgenCounter*/
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "G", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: required element \"G\" missed\n");
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    if(xmlSecNodeGetBNValue(cur, &(key->dsa->g)) == NULL) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to convert element \"G\" value\n");
#endif	    
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    cur = xmlSecGetNextElementNode(cur->next);

    if((cur != NULL) && (xmlSecCheckNodeName(doc, cur, BAD_CAST "X", xmlDSigNs))) {
        /* next is X node. It is REQUIRED for private key but
	 * we are not sure exactly what do we read */
	if(xmlSecNodeGetBNValue(cur, &(key->dsa->priv_key)) == NULL) {
#ifdef DEBUG_XMLSEC    
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyRead: failed to convert element \"X\" value\n");
#endif	    
	    xmlSecKeyDestroy((xmlSecKeyPtr)key);
	    return(NULL);
	}
	key->privateKey = 1;
	cur = xmlSecGetNextElementNode(cur->next);
    }

    /* next is Y node. */
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "Y", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: required element \"Y\" missed\n");
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    if(xmlSecNodeGetBNValue(cur, &(key->dsa->pub_key)) == NULL) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to convert element \"Y\" value\n");
#endif	    
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    cur = xmlSecGetNextElementNode(cur->next);
    
    /* todo: add support for seed */
    if((cur != NULL) && (xmlSecCheckNodeName(doc, cur, BAD_CAST "Seed", xmlDSigNs))) {
	cur = xmlSecGetNextElementNode(cur->next);  
    }

    /* todo: add support for pgencounter */
    if((cur != NULL) && (xmlSecCheckNodeName(doc, cur, BAD_CAST "PgenCounter", xmlDSigNs))) {
	cur = xmlSecGetNextElementNode(cur->next);  
    }

    if(cur != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyRead: unexpected node found\n");
	xmlSecKeyDestroy((xmlSecKeyPtr)key);
	return(NULL);
    }
    return(key);
}

/**
 * xmlSecDsaKeyWrite:
 * @key:	the DSA key
 * @doc:	the xml document to write key to
 * @parent:	the parent DSAKeyValue node 
 * @writePrivateKey: if this flag is 1 then the private
 *		key will be written, otherwise only public
 *		key is written
 *
 * Writes the DSA public or private key into the XML document
 *
 * Returns 0 for success or -1 if an error occurs
 */
int
xmlSecDsaKeyWrite(const xmlSecDsaKeyPtr key, xmlDocPtr doc, xmlNodePtr parent, 
		 int writePrivateKey) {
    xmlNodePtr cur;
    xmlNodePtr prev;
    int ret;
    
    if((key == NULL) || (parent == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyWrite: key or parent is null \n");
#endif 	    
	return(-1);	
    }        
    cur = xmlSecGetNextElementNode(parent->children);
    
    /* first is P node */
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "P", xmlDSigNs))) {
	cur = xmlSecNewSibling(parent, cur, NULL,  BAD_CAST "P");
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyWrite: failed create new node\n");
#endif 	    
	    return(-1);	
	}
    }
    ret = xmlSecNodeSetBNValue(cur, key->dsa->p, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyRead: failed to convert element \"P\" value\n");
#endif	    
	return(-1);
    }    
    prev = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is Q node. */
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur,  BAD_CAST "Q", xmlDSigNs))) {
	cur = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "Q");
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC 
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyWrite: failed create new node\n");
#endif 	    
	    return(-1);	
	}
    }
    ret = xmlSecNodeSetBNValue(cur, key->dsa->q, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyWrite: failed to convert element \"Q\" value\n");
#endif	    
	return(-1);
    }
    prev = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is G node. */
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "G", xmlDSigNs))) {
	cur = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "G");
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyWrite: failed create new node\n");
#endif 	    
	    return(-1);	
	}
    }
    ret = xmlSecNodeSetBNValue(cur, key->dsa->g, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyWrite: failed to convert element \"G\" value\n");
#endif	    
	return(-1);
    }
    prev = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is X node, for private key we will write it,
     otherwise delete */
    if(writePrivateKey) {
	if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "X", xmlDSigNs))) {
	    cur = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "X");
	    if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    		xmlGenericError(xmlGenericErrorContext,
	    	    "xmlSecDsaKeyWrite: failed create new node\n");
#endif 	    
		return(-1);	
	    }
	}
	ret = xmlSecNodeSetBNValue(cur, key->dsa->priv_key, 1);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyWrite: failed to convert element \"X\" value\n");
#endif	    
	    return(-1);
	}
	prev = cur;
	cur = xmlSecGetNextElementNode(cur->next);
    } else if((cur != NULL) && xmlSecCheckNodeName(doc, cur, BAD_CAST "X", xmlDSigNs)) {
	xmlNodePtr ptr;
		
	/* if it is public key remove "X" */
	ptr = cur;
	cur = xmlSecGetNextElementNode(cur->next);
	xmlUnlinkNode(ptr);
	xmlFreeNode(ptr);
    }

    /* next is Y node. */
    if((cur == NULL) || (!xmlSecCheckNodeName(doc, cur, BAD_CAST "Y", xmlDSigNs))) {
	cur = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "Y");
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaKeyWrite: failed create new node\n");
#endif 	    
	    return(-1);	
	}
    }
    ret = xmlSecNodeSetBNValue(cur, key->dsa->pub_key, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaKeyWrite: failed to convert element \"Y\" value\n");
#endif	    
	return(-1);
    }
    prev = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* todo: add support for sed and pgenCounter */
    /* remove the rest */
    while(cur != NULL) {
	prev = cur;
	cur = xmlSecGetNextElementNode(cur->next);
	xmlUnlinkNode(prev);
	xmlFreeNode(prev);
    }    
    return(0);   
}

/**
 *  DSA-SHA1 binary transform
 */

/**
 * xmlSecDsaSha1ContextCalculate:
 * @ctx:	the DSA-SHA1 context
 * @sign:	the flag: sign or verify
 * 
 * Does final sign/verify steps: calls SHA1_Update() and 
 * DSA_do_sign() or DSA_verify()
 *
 * Returns 0 for success (signature created or successfully
 * verified) or -1 if an error occurs or signature does not
 * match.
 */
static int
xmlSecDsaSha1ContextCalculate(xmlSecDsaSha1ContextPtr ctx, int sign) {
    unsigned char buf[SHA_DIGEST_LENGTH];
    int ret;

    if((ctx == NULL) || (ctx->dsa == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1ContextCalculate: ctx is null\n");
#endif	    
	return(-1);
    }
    
    /* first of all finalizae SHA1 */
    SHA1_Final(buf, &(ctx->sha1));

    if(sign) {
	DSA_SIG* sig;
	int rSize, sSize;

	sig = DSA_do_sign(buf, SHA_DIGEST_LENGTH,  ctx->dsa);
	if(sig == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1ContextCalculate: DSA sign failed\n");
#endif 	    
	    return(-1);	    
	}
	rSize = BN_num_bytes(sig->r);
	sSize = BN_num_bytes(sig->s);
	if((rSize > XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE) ||
	   (sSize > XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE)) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1ContextCalculate: r or s is greate than expected %d\n", XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE);
#endif 	    
	    DSA_SIG_free(sig);
	    return(-1);
	}	
	memset(ctx->digest, 0, sizeof(ctx->digest));
	BN_bn2bin(sig->r, ctx->digest + XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE - rSize);
	BN_bn2bin(sig->s, ctx->digest + 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE - sSize);
	DSA_SIG_free(sig);
    } else {
	DSA_SIG* sig;   
	
	sig = DSA_SIG_new();
	if(sig == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1ContextCalculate: failed to create DSA_SIG\n");
#endif 	    
	    return(-1);
	}
	sig->r = BN_bin2bn(ctx->digest, XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE, NULL);
	sig->s = BN_bin2bn(ctx->digest + XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE, XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE, NULL);
	if((sig->r == NULL) || (sig->s == NULL)) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1ContextCalculate: failed to load DSA_SIG from digest\n");
#endif 	    
	    sig = DSA_SIG_new(); 
	    return(-1);
	}
	
	ret = DSA_do_verify(buf, SHA_DIGEST_LENGTH, sig, ctx->dsa);
	DSA_SIG_free(sig); 
	if(ret != 1) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1ContextCalculate: DSA verify failed \n");
#endif 	    
	    return(-1);
	} 
    }
    return(0);
}    

/**
 * xmlSecDsaSha1TransformRead:
 * @ptr:	the DSA-SHA1 trasnform
 * @buf:	the output buffer
 * @len:	the output buffer size
 *
 * Reads data from previous tranform unless there is no more data and
 * calculates SHA1 digest. After this the digest is signed/verified using 
 * DSA1 and result returned in the output buffer.
 * 
 * Returns the number of bytes written to output buffer or 
 * -1 if an error occurs.  
 */
static int
xmlSecDsaSha1TransformRead(xmlSecBinTransformPtr ptr, unsigned char *buf, size_t len) {
    xmlSecDsaSha1ContextPtr ctx;
    int ret;

    if((ptr == NULL) || (ptr->data == NULL) || (buf == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformDestroy: ptr, ptr->data or buf is null\n");
#endif	    
	return(-1);
    }

    /* if we already called Final then nothing to read more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (xmlSecDsaSha1ContextPtr)(ptr->data);
    /* have to read everything first */
    do {
	ret = xmlSecBinTransformRead(ptr->prev, buf, len);
        if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSha1TransformRead: prev read failed\n");
#endif 	    
	    return(-1);
	} else if(ret > 0) {
	    SHA1_Update(&(ctx->sha1), buf, ret);
	}
    } while(ret > 0);
    
    /* ret == 0 i.e. there is no more data */
    ptr->finalized = 1;
    ret = xmlSecDsaSha1ContextCalculate(ctx, ptr->encode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformDestroy: digest calculation failed\n");
#endif	    
	return(-1);
    }
    

    if(len < 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformDestroy: too small\n");
#endif	    
	return(-1);
    }
    
    memcpy(buf, ctx->digest, 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE);
    return(2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE);
}

/**
 * xmlSecDsaSha1TransformWrite:
 * @ptr:	the DSA-SHA1 transform
 * @buf:	the input buffer
 * @len:	the input buffer len
 *
 * Updates the SHA1 digest with new data. No data are written
 * to the next buffer
 *
 * Returns the number of processed bytes (len) or -1 if an error occurs.
 */
static int
xmlSecDsaSha1TransformWrite(xmlSecBinTransformPtr ptr, const unsigned char *buf, size_t len) {
    xmlSecDsaSha1ContextPtr ctx;

    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    if((buf == NULL) || (len == 0)) {
	return(0);
    }

    /* if we already called Final then nothing to write more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (xmlSecDsaSha1ContextPtr)(ptr->data);
    SHA1_Update(&(ctx->sha1), buf, len);
    return(len);
}


/**
 * xmlSecDsaSha1TransformFlush:
 * @ptr:	the DSA-SHA1 transform
 *
 * Calculates the SHA1 digest, signs/verifies it using DSA,
 * writes results to the next transform and calls its flush.
 *
 * Returns the number of bytes written to the next transform
 * or -1 if an error occurs or signature does not match.
 */
static int
xmlSecDsaSha1TransformFlush(xmlSecBinTransformPtr ptr) {
    xmlSecDsaSha1ContextPtr ctx;
    int ret;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformFlush: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    /* if we already called Final then nothing to flush more! */
    if(ptr->finalized) {
	return(0);
    }

    ctx = (xmlSecDsaSha1ContextPtr)(ptr->data);
    ret = xmlSecDsaSha1ContextCalculate(ctx, ptr->encode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformFlush: digest calculation failed\n");
#endif	    
	return(-1);
    }
    
    ret = xmlSecBinTransformWrite(ptr->next, ctx->digest, 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecSha1TransformFlush: prev write failed\n");
#endif 	    
	return(-1);
    }

    ret = xmlSecBinTransformFlush(ptr->next);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformFlush: prev flush failed\n");
#endif 	    
	return(-1);
    } 
    return(0);
}

/**
 * xmlSecDsaSha1TransformDestroy:
 * @ptr:	the DSA-SHA1 trasnform
 *
 * Destroys DSA-SHA1 transform and frees allocated memory
 */
static void
xmlSecDsaSha1TransformDestroy(xmlSecBinTransformPtr ptr) {
    xmlSecDsaSha1ContextPtr ctx; 

    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformDestroy: ptr is null\n");
#endif	    
	return;
    }

    if(ptr->data == NULL) {
	xmlFree(ptr);
	return;
    }

    ctx = (xmlSecDsaSha1ContextPtr)(ptr->data);
    if(ctx->dsa != NULL) {
	DSA_free(ctx->dsa);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform) + sizeof(xmlSecDsaSha1Context));
    xmlFree(ptr);
}

/**
 * xmlSecDsaSha1TransformCreate:
 * @encode:	the encode(1) or decode(0) flag
 * @dsaKey:	the dsa key to use (transform creates a new copy if this key)
 * @digest:	the digest to verify (if @encode equal to 1 then this parameter
 *		ignored)
 *
 * Creates new DSA-SHA1 trasnform (copies the key, etc).
 *
 * Returns newly allocated DSA-SHA1 trasnform (caller is responsible for
 * deleting it using xmlSecTransformDestroy() function) or NULL if an error
 * occurs.
 */
xmlSecBinTransformPtr
xmlSecDsaSha1TransformCreate(int encode, xmlSecDsaKeyPtr dsaKey, const xmlChar* digest) {
    xmlSecBinTransformPtr ptr;
    xmlSecDsaSha1ContextPtr ctx;
    int ret;

    if((dsaKey == NULL) || (dsaKey->dsa == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: dsaKey is null\n");
#endif	    
	return(NULL);
    }
    /*
     * Allocate a new xmlSecBinTransform  + xmlSecDsaSha1Context + 
     * space for result md and fill the fields.
     */
    ptr = (xmlSecBinTransformPtr)xmlMalloc(sizeof(xmlSecBinTransform) + sizeof(xmlSecDsaSha1Context));
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform) + sizeof(xmlSecDsaSha1Context));

    ctx = (xmlSecDsaSha1ContextPtr)(((unsigned char*)ptr) + sizeof(xmlSecBinTransform));

    ptr->algorithm = xmlSecSignDsaSha1;
    ptr->encode = encode;
    ptr->data = ctx;
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecDsaSha1TransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecDsaSha1TransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecDsaSha1TransformWrite;
    ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecDsaSha1TransformFlush;
    
    /* 
     * decode digest
     */
    if((!encode) && (digest != NULL)) {
	ret = xmlSecBase64Decode(digest, ctx->digest, 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE);
	if(ret < 0) {	
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1TransformCreate: failed to base64 decode digest\n");
#endif 	    
	    xmlSecDsaSha1TransformDestroy(ptr);	
	    return(NULL);	    
	}	
	if(ret < 2 * XMLSEC_DSA_SHA1_HALF_DIGEST_SIZE) {
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDsaSha1TransformCreate: digest is too small\n");
	    xmlSecDsaSha1TransformDestroy(ptr);	
	    return(NULL);	    
	}
    }    

    /* 
     * create SHA1
     */
    SHA1_Init(&(ctx->sha1));
     
    /* 
     * now create DSA context and copy all keys. 
     * todo: do we need this or we can 
     * reuse existing structure?
     */
    ctx->dsa = DSA_new();
    if(ctx->dsa == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create DSA context\n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   
    if((dsaKey->dsa != NULL) && (dsaKey->dsa->p != NULL)) {
	ctx->dsa->p = BN_dup(dsaKey->dsa->p);
    } else {
	ctx->dsa->p = BN_new();
    }
    if(ctx->dsa->p == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create p \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   

    if((dsaKey->dsa != NULL) && (dsaKey->dsa->p != NULL)) {
	ctx->dsa->p = BN_dup(dsaKey->dsa->p);
    } else {
	ctx->dsa->p = BN_new();
    }
    if(ctx->dsa->p == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create p \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   

    if((dsaKey->dsa != NULL) && (dsaKey->dsa->q != NULL)) {
	ctx->dsa->q = BN_dup(dsaKey->dsa->q);
    } else {
	ctx->dsa->q = BN_new();
    }
    if(ctx->dsa->q == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create q \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   

    if((dsaKey->dsa != NULL) && (dsaKey->dsa->g != NULL)) {
	ctx->dsa->g = BN_dup(dsaKey->dsa->g);
    } else {
	ctx->dsa->g = BN_new();
    }
    if(ctx->dsa->g == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create g \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   

    if((dsaKey->dsa != NULL) && (dsaKey->dsa->pub_key != NULL)) {
	ctx->dsa->pub_key = BN_dup(dsaKey->dsa->pub_key);
    } else {
	ctx->dsa->pub_key = BN_new();
    }
    if(ctx->dsa->pub_key == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create pub_key \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   

    if((dsaKey->dsa != NULL) && (dsaKey->dsa->priv_key != NULL)) {
	ctx->dsa->priv_key = BN_dup(dsaKey->dsa->priv_key);
    } else {
	ctx->dsa->priv_key = BN_new();
    }
    if(ctx->dsa->priv_key == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDsaSha1TransformCreate: failed to create priv_key \n");
#endif 	    
	xmlSecDsaSha1TransformDestroy(ptr);
	return(NULL);
    }   
    
    return(ptr);
}

