/** 
 * XMLSec library
 *
 * HMAC algorithm support
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/hmac.h>
#include <libxml/xmlerror.h> 

#include <xmlsec/xmlsec.h>
#include <xmlsec/algorithms.h>
#include <xmlsec/base64.h>
#include <xmlsec/hmac.h>

/**
 * HMAC context
 */
typedef struct _xmlSecHmacCtx {
    HMAC_CTX		ctx;
    size_t		mdLenRequested;
    unsigned char	md[EVP_MAX_MD_SIZE];
    unsigned int	mdLen;
} xmlSecHmacCtx, *xmlSecHmacCtxPtr;

/**
 * HMAC key
 */

/**
 * xmlSecHmacKeyCreate:
 * @name:	the key name
 * @key:	the key
 * @keyLen:	the key length
 *
 * Creates new HMAC key.
 *
 * Returns the HMAC key (caller is reponsible for freeing the memory
 * using xmlSecKeyDestroy() function) or NULL an error occurs
 */
xmlSecHmacKeyPtr	
xmlSecHmacKeyCreate(const xmlChar* name, const unsigned char *key, size_t keyLen) {
    xmlSecHmacKeyPtr ptr;
    size_t size;
    
    /*
     * Allocate a new xmlSecHmacKey
     */
    size = sizeof(xmlSecHmacKey) + (sizeof(unsigned char) * keyLen);
    ptr = (xmlSecHmacKeyPtr) xmlMalloc(size);
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacKeyCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }    
    memset(ptr, 0, size);
    
    ptr->name = (name != NULL) ? xmlStrdup(name) : NULL;
    ptr->algorithm = xmlSecMacHmacSha1;
    ptr->size = size;
    ptr->keyLen = keyLen;
    ptr->privateKey = (key != NULL) ? 1 : 0;
    if(key != NULL) {
	memcpy(ptr->key, key, keyLen);
    }
    return(ptr);
}

/**
 * xmlSecHmacKeyRead:
 * @doc:	the container XML document
 * @hmacKeyValueNode:	the KeyValue node with HMAC key
 * @name:	the key name
 *
 * Reads HMAC key from XML document.
 *
 * Returns the HMAC key (caller is reponsible for freeing the memory
 * using xmlSecKeyDestroy() function) or NULL an error occurs
 */
xmlSecHmacKeyPtr
xmlSecHmacKeyRead(const xmlDocPtr doc, const xmlNodePtr hmacKeyValueNode, 
		  const xmlChar* name) {
    xmlSecHmacKeyPtr key = NULL;
    xmlChar* content;
    unsigned char* buf;
    size_t size;
    int ret;
    
    if(hmacKeyValueNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHMACKeyRead: hmacKeyValueNode is null\n");
#endif 	    
	return(NULL);
    }    

    content = xmlNodeGetContent(hmacKeyValueNode);
    if(content == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHMACKeyRead: failed to get content\n");
#endif 	    
	return(NULL);	
    }
    size = (3 * xmlStrlen(content)) / 4 + 3;
    
    buf = (unsigned char*)xmlMalloc(size + 1);
    if(buf == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHMACKeyRead: failed to create buffer of %d bytes\n", size);
#endif 	    
	xmlFree(content);
	return(NULL);	
    }
    
    ret = xmlSecBase64Decode(content, buf, size);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHMACKeyRead: failed to base64 decode content\n");
#endif 	    
	xmlFree(buf);
	xmlFree(content);
	return(NULL);	
    }
    
    key = xmlSecHmacKeyCreate(name, buf, ret);
    if(key == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHMACKeyRead: failed to create key\n");
#endif 	    
	xmlFree(buf);
	xmlFree(content);
	return(NULL);	
    }
    
    xmlFree(buf);
    xmlFree(content);    
    return(key);
}

/**
 * xmlSecHmacKeyWrite:
 * @key:	the HMAC key
 * @doc:	the container XML document
 * @parent:	the parent KeyValue node
 * @writePrivateKey: the flag write private key or not.
 *
 * Writes the HMAC key into an XML document
 *
 * Returns 0 on success and -1 otherwise.
 */
int
xmlSecHmacKeyWrite(const xmlSecHmacKeyPtr key, xmlDocPtr doc, 
		 xmlNodePtr parent, int writePrivateKey) {
    xmlChar* content;
    
    if((key == NULL) || (parent == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacKeyWrite: key or parent is null \n");
#endif 	    
	return(-1);	
    }        

    if(!writePrivateKey) {
	return(0);
    }
        
    content = xmlSecBase64Encode(key->key, key->keyLen, XMLSEC_BASE64_LINESIZE);
    if(content == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacKeyWrite: failed to base64 encode the key \n");
#endif 	    
	return(-1);	
    }
    
    xmlNodeSetContent(parent, content);
    xmlFree(content);
    return(0);
}		 
		


/**
 * HMAC binary transform
 */
		 
/**
 * xmlSecHmacTransformRead:
 * @ptr: 	the HMAC transform
 * @buf:	the pointer to output buffer
 * @len:	the output buffer size
 *
 * Reads data from previous tranform unless no more data available
 * and calculates the digest. After this the result digest is returned.
 * 
 * Returns the number of bytes written to output buffer or 
 * -1 if an error occurs. 
 */
static int
xmlSecHmacTransformRead(xmlSecBinTransformPtr ptr, unsigned char *buf, size_t len) {
    int ret;
    xmlSecHmacCtxPtr ctx;     
       
    if((ptr == NULL) || (buf == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformRead: ptr, ptr->data or buffer is null\n");
#endif	    
	return(-1);
    }
    
    ctx = (xmlSecHmacCtxPtr)ptr->data;
    if(len < ctx->mdLenRequested) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformRead: input len %d is not enough (%d bytes required)\n", len, ctx->mdLenRequested);
	return(-1);
    }
    

    /* if we already called Final then nothing to read more! */
    if(ptr->finalized) {
	return(0);
    }
    
    /* have to read everything first */
    do {
	ret = xmlSecBinTransformRead(ptr->prev, buf, len);
        if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecHmacTransformRead: prev read failed\n");
#endif 	    
	    return(-1);
	} else if(ret > 0) {
	    HMAC_Update(&ctx->ctx, buf, ret);	
	}
    } while(ret > 0);

    /* ret == 0 i.e. there is no more data */
    HMAC_Final(&ctx->ctx, buf, &ctx->mdLen);
    ptr->finalized = 1;

    /* truncate data */
    if((ctx->mdLen > ctx->mdLenRequested) && (ctx->mdLenRequested != 0)) {
	ctx->mdLen = ctx->mdLenRequested;    
    }
    return(ctx->mdLen);
}

/**
 * xmlSecHmacTransformWrite:
 * @ptr:	the HMAC transform
 * @buf:	the input buffer
 * @len:	the input buffer len
 *
 * Updates the HMAC digest with new data. No data are written
 * to the next buffer
 *
 * Returns the number of processed bytes (len) or -1 if an error occurs.
 */
static int
xmlSecHmacTransformWrite(xmlSecBinTransformPtr ptr, const unsigned char *buf, size_t len) {
    xmlSecHmacCtxPtr ctx;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }
    
    if((buf == NULL) || (len == 0)) {
	return(0);
    }

    /* if we already called Final then nothing to write more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (xmlSecHmacCtxPtr)ptr->data;     
    HMAC_Update(&ctx->ctx, buf, len);
    return(len);
}


/**
 * xmlSecHmacTransformFlush:
 * @ptr:	the HMACtransform
 *
 * Calculates the HMAC digest, writes results to the next transform and 
 * calls its flush.
 *
 * Returns the number of bytes written to the next transform
 * or -1 if an error occurs or signature does not match.
 */
static int
xmlSecHmacTransformFlush(xmlSecBinTransformPtr ptr) {
    int ret;
    xmlSecHmacCtxPtr ctx;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformFlush: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    /* if we already called Final then nothing to flush more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (xmlSecHmacCtxPtr)ptr->data; 
    HMAC_Final(&ctx->ctx, ctx->md, &ctx->mdLen); 
    if(ctx->mdLen > ctx->mdLenRequested) ctx->mdLen = ctx->mdLenRequested;
        
    ret = xmlSecBinTransformWrite(ptr->next, ctx->md, ctx->mdLen);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformFlush: prev write failed\n");
#endif 	    
	return(-1);
    } 
    
    
    ret = xmlSecBinTransformFlush(ptr->next);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformFlush: prev flush failed\n");
#endif 	    
	return(-1);
    } 
    return(0);
}
 
/**
 * xmlSecHmacTransformDestroy:
 * @ptr:	the HMAC transforms
 *
 * Destroys HMAC transform ad frees the allocated memory
 */
static void 
xmlSecHmacTransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    /* purge memory */
    xmlFree(ptr);
}

/**
 * xmlSecHmacOutputLengthRead:
 * @cur:	the Trasnform node for HMAC
 *
 * Reads the HMACOutputLength node.
 *
 * Returns the length or -1 if an error occurs or 
 * there is no HMACOutputLength child node.
 */
static int
xmlSecHmacOutputLengthRead(xmlNodePtr cur) {
    xmlChar* content;
    int ret;
    
    if(cur == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacOutputLengthRead: ptr or cur is null\n");	
#endif
	return(-1);
    }

    cur = xmlSecGetNextElementNode(cur->children);
    while((cur != NULL) && (!xmlStrEqual(cur->name, BAD_CAST "HMACOutputLength"))) {
	cur = xmlSecGetNextElementNode(cur->next);
    }
    if(cur == NULL)  {
	return(-1);
    }
    content = xmlNodeGetContent(cur);
    if(content == NULL) {
	return(-1);
    }
    ret = atoi((char*)content);
    xmlFree(content);
    return(ret / 8); /* content length is in bits */
}

/**
 * xmlSecHmacTransformCreate:
 * @transformNode:	the Trasnform node
 * @hmacKey:		the HMAC key
 *
 * Creates new HMAC transform and reads any necessary additional data
 * from the Transform node.
 *
 * Returns new HMAC transform (caller is responsible for deleting it using
 * xmlSecBinTransformDestroy() function) or NULL if an error occurs.
 */
xmlSecBinTransformPtr	
xmlSecHmacTransformCreate (xmlNodePtr transformNode, xmlSecHmacKeyPtr hmacKey) {
    xmlSecBinTransformPtr ptr;
    xmlSecHmacCtxPtr ctx;
    size_t size;
        
    if(hmacKey == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformCreate: hmacKey is null\n");
#endif	    
	return(NULL);
    }
    
    /*
     * Allocate a new xmlSecBinTransform  + xmlSecHmacCtx + 
     * space for result md and fill the fields.
     */
    size = sizeof(xmlSecBinTransform) + sizeof(xmlSecHmacCtx);
    ptr = (xmlSecBinTransformPtr) xmlMalloc(size);
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecHmacTransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, size);

    /* The HMAC algorithm (RFC2104 [HMAC]) takes the truncation length in 
     * bits as a parameter; if the parameter is not specified then all the 
     * bits of the hash are output.*/
    

    ctx = (xmlSecHmacCtxPtr)(((unsigned char*)ptr) + sizeof(xmlSecBinTransform));
    
    ptr->algorithm = xmlSecMacHmacSha1;
    ptr->encode = 1;
    ptr->data = ctx;
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecHmacTransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecHmacTransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecHmacTransformWrite;
    ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecHmacTransformFlush;

    if(transformNode != NULL) {
	int len;
	
	len = xmlSecHmacOutputLengthRead(transformNode);   
	ctx->mdLenRequested = (len > 0) ? (size_t)len :  EVP_MAX_MD_SIZE;	
    } else {
	ctx->mdLenRequested = EVP_MAX_MD_SIZE;
    }

    /* now create hmac context */ 
    HMAC_Init(&ctx->ctx, hmacKey->key, hmacKey->keyLen, EVP_sha1());
    
    return(ptr);    
}

