/** 
 * XMLSec library
 *
 * Memory buffer and IO transforms
 *
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <libxml/nanohttp.h>
#include <libxml/nanoftp.h>

#include <xmlsec/io.h>

/**
 * Memory buffer binary transform
 */
/**
 * xmlSecBufferTransformRead:
 * @ptr:	the Memory Buffer Trasnform
 * @buf:	the output buffer
 * @len:	the output buffer length
 *
 * Returns data from the memory buffer to the caller.
 *
 * Returns the size of data returned or -1 if an error
 * occurs
 */
static int
xmlSecBufferTransformRead(xmlSecBufferTransformPtr ptr, unsigned char *buf, size_t len) {
    xmlBufferPtr buffer;
    
    if((ptr == NULL) || (buf == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformRead: ptr, ptr->data or buffer is null\n");
#endif	    
	return(-1);
    }
    
    buffer = (xmlBufferPtr)(ptr->data);
    if((xmlBufferLength(buffer) > 0) && (len > (size_t)xmlBufferLength(buffer))) {
	len = (size_t)xmlBufferLength(buffer);
    }
    if(len > 0) {
	/* copy data to the caller */
	memcpy(buf, xmlBufferContent(buffer), len);
	/* remove them from our buffer */
	xmlBufferShrink(buffer, len);
    }
    return(len);
}

/**
 * xmlSecBufferTransformWrite:
 * @ptr:	the memory buffer transform
 * @buf:	the input buffer	
 * @len:	the input buffer length
 *
 * Saves data into the memory buffer
 *
 * Returns the size of data saved or -1 if an error occurs.
 */
static int
xmlSecBufferTransformWrite(xmlSecBufferTransformPtr ptr, const unsigned char *buf, size_t len) {

    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }
    if(buf == NULL || len == 0) {
	return(0);
    }
    xmlBufferAdd((xmlBufferPtr)(ptr->data), buf, len);
    return(len);
}

/**
 * xmlSecBufferTransformDestroy:
 * @ptr:	the memory buffer transform
 *
 * Destroyes the memory buffer transform
 */
static void
xmlSecBufferTransformDestroy(xmlSecBufferTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    if(ptr->data != NULL) {
	xmlBufferEmpty((xmlBufferPtr)(ptr->data));
	xmlBufferFree((xmlBufferPtr)(ptr->data)); 
    }    
    xmlFree(ptr);
}
 
/**
 * xmlSecBufferTransformCreate:
 * 
 * Creates new memory buffer trasnform
 *
 * Returns newly allocated memory buffer transform (caller is responsible
 * for freeing memory with xmlSecTrasnformDestroy() function) or NULL if an
 * error occurs.
 */
xmlSecBufferTransformPtr 
xmlSecBufferTransformCreate(void) {
    xmlSecBufferTransformPtr ptr;

    /*
     * Allocate a new xmlSecBufferTransform and fill the fields.
     */
    ptr = (xmlSecBufferTransformPtr)xmlMalloc(sizeof(xmlSecBinTransform));
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform));
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecBufferTransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecBufferTransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecBufferTransformWrite;

    ptr->data = xmlBufferCreate();
    if(ptr->data == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformCreate: xmlBufferCreate failed\n");
#endif 	    
	xmlSecBufferTransformDestroy(ptr);	
	return(NULL);
    }
    
    return(ptr);    
}

/**
 * xmlSecBufferTransformGet:
 * @ptr: 	the memory buffer transform
 *
 * Returns the memory buffer.
 *
 * Returns the memory buffer or NULL if an error occurs.
 */
xmlBufferPtr		 
xmlSecBufferTransformGet(xmlSecBufferTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBufferTransformGet: ptr is null\n");
#endif	    
	return(NULL);
    }
    return((xmlBufferPtr)(ptr->data));
}


/* 
 * IO binary transform 
 */
/**
 * xmlSecUriTransformRead:
 * @ptr:	the IO transform
 * @buf:	the input buffer
 * @len:	the input buffer length
 *
 * Reads the data from local file, http or ftp.
 *
 * Returns the size of returned data of -1 if an error occurs.
 */
static int
xmlSecUriTransformRead(xmlSecBinTransformPtr ptr, unsigned char *buf, size_t len) {
    int ret;
        
    if((ptr == NULL) || (buf == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformRead: ptr, ptr->data or buffer is null\n");
#endif	    
	return(-1);
    }

    switch(ptr->algorithm) {
    case xmlSecInputLocalFile:
	ret = fread(buf, sizeof(unsigned char), len, (FILE*)ptr->data);
	break;
    case xmlSecInputHttpFile:
	ret = xmlNanoHTTPRead(ptr->data, buf, len);
	break;
    case xmlSecInputFtpFile:
	ret = xmlNanoFTPRead(ptr->data, buf, len);
        break;
    default:
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformRead: unsupported algorithm %d\n", ptr->algorithm);
	return(-1);
    }

    if(ret < 0) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformRead: read failed (errno=%d)\n", errno);
	return(-1);
    } 
    return(ret);
}

/**
 * xmlSecUriTransformWrite:
 * @ptr:	the IO trasnform
 * @buf:	the output buffer
 * @len:	the output buffer length
 *
 * Writes data to local file
 *
 * Returns the nnumber of bytes written on -1 if an error occurs.
 */
static int
xmlSecUriTransformWrite(xmlSecBinTransformPtr ptr, const unsigned char *buf, size_t len) {
    int ret;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }
    
    if((buf == NULL) || (len == 0)) {
	return(0);
    }

    switch(ptr->algorithm) {
    case xmlSecInputLocalFile:
	ret = fwrite(buf, sizeof(unsigned char), len, (FILE*)ptr->data);
	break;
    default:
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformWrite: unsupported algorithm %d\n", ptr->algorithm);
	return(-1);
    }

    if(ret < 0) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformWrite: write() failed (errno=%d)\n", errno);
	return(-1);
    } 
    return(ret);
}

/**
 * xmlSecUriTransformFlush:
 * @ptr:	the IO trasnform  
 *
 * Flushes data to the file
 *
 * Returns 0 for success and -1 if an error occurs.
 */
static int
xmlSecUriTransformFlush(xmlSecBinTransformPtr ptr) {
    int ret;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformFlush: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    switch(ptr->algorithm) {
    case xmlSecInputLocalFile:
	ret = fflush((FILE*)ptr->data);
	break;
    default:
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformFlush: unsupported algorithm %d\n", ptr->algorithm);
	return(-1);
    }

    if(ret < 0) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformFlush: flush() failed (errno=%d)\n", errno);
	return(-1);
    } 
    return(0);
}
 
/**
 * xmlSecUriTransformDestroy:
 * @ptr:	the IO transform
 *
 * Destroys IO trasnform
 *
 */
static void 
xmlSecUriTransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformDestroy: ptr is null\n");
#endif	    
	return;
    }    
    if(ptr->data != NULL) {
	switch(ptr->algorithm) {
	case xmlSecInputLocalFile:
    	    fclose((FILE*)ptr->data);
	    break;
	case xmlSecInputHttpFile:
	    xmlNanoHTTPClose(ptr->data);
	    break;
	case xmlSecInputFtpFile:
	    xmlNanoFTPClose(ptr->data);
	    break;
	default:
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecUriTransformDestroy: unsupported algorithm %d\n", ptr->algorithm);
	}
    }    
    xmlFree(ptr);
}
 
 
/**
 * xmlSecUriTransformCreate:
 * @uri:	the uri to read or write
 * @read:	the operation flag (1 - read, 0 - write)
 *
 * Creates new IO transform (opens file, etc.)
 * 
 * Returns newly allocated trasnform (caller is responsible for deleting it) or
 * NULL if an error occurs.
 */
xmlSecBinTransformPtr 
xmlSecUriTransformCreate(const char* uri, int read) {
    xmlSecBinTransformPtr ptr;

    /*
     * Allocate a new xmlSecUriTransform and fill the fields.
     */
    ptr = (xmlSecBinTransformPtr) xmlMalloc(sizeof(xmlSecBinTransform));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform));

    ptr->encode = read;    
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecUriTransformDestroy;
    if(read) {
        ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecUriTransformRead;
    } else {
	ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecUriTransformWrite;
	ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecUriTransformFlush;
    }

    /* todo: add an ability to use custom protocol handlers */
    if(strncmp(uri, "http://", 7) == 0) {
	ptr->algorithm = xmlSecInputHttpFile;
	ptr->data = xmlNanoHTTPOpen(uri, NULL);
    } else if(strncmp(uri, "ftp://", 6) == 0) { 
	ptr->algorithm = xmlSecInputFtpFile;
	ptr->data = xmlNanoFTPOpen(uri);
    } else {
	FILE *fd;
	const char *path = NULL;
	
	ptr->algorithm = xmlSecInputLocalFile;
	/* try to open local file */
	if(strncmp(uri, "file://localhost", 16) == 0) {
	    path = &uri[16];
	} else if(strncmp(uri, "file:///", 8) == 0) {
#if defined (_WIN32) && !defined(__CYGWIN__)
	    path = &uri[8];
#else
	    path = &uri[7];
#endif
	} else {
	    path = uri;
	}
#if defined(WIN32) || defined (__CYGWIN__)
	fd = fopen(path, (read) ? "rb" : "wb");
#else
	fd = fopen(path, (read) ? "r" : "w");
#endif /* WIN32 */
	ptr->data = fd;
    }

    if(ptr->data == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecUriTransformCreate: unable to open file \"%s\"\n", uri);
	xmlSecUriTransformDestroy(ptr);
	return(NULL);
    }

    return(ptr);    
}


/*
 * debug transform: dump data to a file and pass to next transform 
 */
/**
 * xmlSecDebugDumpTransformRead:
 * @ptr:	the debug dump trasnform
 * @buf:	the output buffer
 * @len:	the output buffer size
 * 
 * Reads data from previous transform, writes to the file and returns
 * to the caller.
 *
 * Returns the number of bytes returned or -1 if an error occurs.
 */
static int
xmlSecDebugDumpTransformRead(xmlSecBinTransformPtr ptr, unsigned char *buf, size_t len) {
    int ret;
        
    if((ptr == NULL) || (buf == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformRead: ptr or buffer is null\n");
#endif	    
	return(-1);
    }
    
    ret = xmlSecBinTransformRead(ptr->prev, buf, len);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformRead: prev read failed\n");
#endif 	    
	return(-1);
    }
    
    if((ret > 0) && (ptr->data != NULL)) {
	fwrite(buf, ret, sizeof(unsigned char), (FILE*)ptr->data);
    } 
    return(ret);
}

/**
 * xmlSecDebugDumpTransformWrite:
 * @ptr:	the debug dump trasnform
 * @buf:	the input buffer
 * @len:	the input buffer size
 *
 * Writes data to the next trasnform and the file.
 *
 * Returns the number of bytes written or -1 if an error
 * occurs.
 */
static int
xmlSecDebugDumpTransformWrite(xmlSecBinTransformPtr ptr, 
		const unsigned char *buf, size_t len) {
    int ret;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformWrite: ptr is null\n");
#endif	    
	return(-1);
    }
    if(buf == NULL) {
	return(0);
    }

    if((len > 0) && (ptr->data != NULL)) {
	fwrite(buf, len, sizeof(unsigned char), (FILE*)ptr->data);
    } 
    
    ret = xmlSecBinTransformWrite(ptr->next, buf, len);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformWrite: prev write failed\n");
#endif 	    
	return(-1);
    }	 
    
    return(ret);
}

/**
 * xmlSecDebugDumpTransformFlush:
 * @ptr:	the debug dump transform
 *
 * Flushes the next transform
 *
 */
static int
xmlSecDebugDumpTransformFlush(xmlSecBinTransformPtr ptr) {
    int ret;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformFlush: ptr is null\n");
#endif	    
	return(-1);
    }

    if(ptr->data != NULL) {
	fflush((FILE*)ptr->data);
    } 

    ret = xmlSecBinTransformFlush(ptr->next);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformFlush: prev flush failed\n");
#endif 	    
	return(-1);
    }     
    
    return(0);
}


/**
 * xmlSecDebugDumpTransformDestroy:
 * @ptr:	the debug dump transform
 *
 * Destroys the debug dump trasnform
 *
 */
static void 
xmlSecDebugDumpTransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    if(ptr->data != NULL) {
	fclose((FILE*)ptr->data);
    }    
    xmlFree(ptr);
}


/**
 * xmlSecDebugDumpTransformCreate:
 * @filename:
 *
 * Creates new debug dump transform. All data comming thru transform
 * will be dumped to the specfied file.
 *
 * Returns the newly created debug dump transform (caller is responsible
 * for destroying it) or NULL if an error occurs.
 */
xmlSecBinTransformPtr		
xmlSecDebugDumpTransformCreate (const char *filename) {
    FILE *fd;
    xmlSecBinTransformPtr ptr;

    if(filename == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformCreate: filename is null\n");
#endif	    
	return(NULL);
    }

#if defined(WIN32) || defined (__CYGWIN__)
    fd = fopen(filename, "wb");
#else
    fd = fopen(filename, "w");
#endif /* WIN32 */

    if(fd == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformCreate: unable to open file \"%s\"\n", filename);
	return(NULL);
    }

    /*
     * Allocate a new xmlSecDebugDumpTransform and fill the fields.
     */
    ptr = (xmlSecBinTransformPtr) xmlMalloc(sizeof(xmlSecBinTransform));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDebugDumpTransformCreate: malloc failed\n");
#endif 	    
	fclose(fd);
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform));

    ptr->data = fd;
    
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecDebugDumpTransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecDebugDumpTransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecDebugDumpTransformWrite;
    ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecDebugDumpTransformFlush;
    return(ptr);    
}


