/** 
 * XMLSec library
 *
 * Keys Mangement
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/tree.h>
#include <libxml/list.h>

#include <xmlsec/xmlsec.h>
#include <xmlsec/base64.h>
#include <xmlsec/hmac.h>
#include <xmlsec/dsa.h>
#include <xmlsec/rsa.h>

#include <xmlsec/keysmgmt.h> 

/**
 * Base Key
 */
/**
 * xmlSecKeyDestroy:
 * @ptr:	the key
 *
 * Destroys the xmlSecKey. First we are trying to call
 * key-specific destroy callback. If it is not available
 * we have nothing else other than xmlFree()
 */
void
xmlSecKeyDestroy(xmlSecKeyPtr ptr) {
    if(ptr == NULL) {
	return;
    }
    if(ptr->name != NULL) {
	xmlFree(ptr->name);
	ptr->name = NULL;
    }
    if(ptr->destroyCallback) {
	ptr->destroyCallback(ptr);
    } else {
	if(ptr->size > 0) {
	    memset(ptr, 0, ptr->size);
	}
	xmlFree(ptr);
    }
}

/**
 * http://www.w3.org/TR/xmlSec-core/#sec-KeyValue
 *
 * The KeyValue  Element
 *
 * The KeyValue element contains a single public key that may be useful in 
 * validating the signature. Structured formats for defining DSA (REQUIRED) 
 * and RSA (RECOMMENDED) public keys are defined in Signature Algorithms 
 * (section 6.4). The KeyValue element may include externally defined public 
 * keys values represented as PCDATA or element types from an external 
 * namespace.
 *
 * Schema Definition:
 *
 * <element name="KeyValue" type="ds:KeyValueType"/> 
 * <complexType name="KeyValueType" mixed="true">
 *   <choice>
 *     <element ref="ds:DSAKeyValue"/>
 *     <element ref="ds:RSAKeyValue"/>
 *     <any namespace="##other" processContents="lax"/>
 *   </choice>
 * </complexType>
 *    
 * DTD:
 *    
 * <!ELEMENT KeyValue (#PCDATA|DSAKeyValue|RSAKeyValue %KeyValue.ANY;)* >
 *       
 */

/**
 * xmlSecKeyValueRead:
 * @doc:	the container XML doc
 * @keyValueNode:	the KeyValue node
 * @name:	the key name
 *
 * Reads key from the XML document KeyValue node and creates a new
 * xmlSecKey object.
 *
 * Returns the newly created object (caller is responsible for destroying it)
 * or NULL if an error occurs.
 */
xmlSecKeyPtr
xmlSecKeyValueRead(const xmlDocPtr doc, const xmlNodePtr keyValueNode, 
		   const xmlChar *name) {
    xmlSecKeyPtr key = NULL;
    xmlNodePtr cur;

    if(keyValueNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueRead: keyValueNode is null\n");	
#endif
	return(NULL);
    }
    
    /* todo: add Retrival and x509 support */
    cur = xmlSecGetNextElementNode(keyValueNode->children);
    while(cur != NULL) {
	if(xmlSecCheckNodeName(doc, cur, BAD_CAST "HMACKeyValue", xmlDSigNs)) {
	    key = (xmlSecKeyPtr)xmlSecHmacKeyRead(doc, cur, name);
	    break;
	} else if(xmlSecCheckNodeName(doc, cur, BAD_CAST "DSAKeyValue", xmlDSigNs)) {
	    key = (xmlSecKeyPtr)xmlSecDsaKeyRead(doc, cur, name);
	    break;
	} else if(xmlSecCheckNodeName(doc, cur, BAD_CAST "RSAKeyValue", xmlDSigNs)) {
	    key = (xmlSecKeyPtr)xmlSecRsaKeyRead(doc, cur, name);
	    break;
	}
	cur = xmlSecGetNextElementNode(cur->next);
    }

    if(key == NULL) {
	xmlGenericError(xmlGenericErrorContext, "xmlSecKeyValueRead: no key value found\n");	
	return(NULL);
    }
    key->node = cur; /* remember node position! */
    return(key);
}

/**
 * xmlSecKeyValueFindNode:
 * @doc:	the container XML doc
 * @keyValueNode:	the KeyValue node
 * @alg:	the key algorithm 
 * @addIfNotFound:	the flag, if set (1) then the key node
 *		would be added to the doc
 *
 * Searches the childs of KeyValue node for the key description.
 * If there is no one and @addIfNotFound flag is set then 
 * a new node is added
 *
 * Returns the pointer to key's node or NULL if node is not found
 * or an error occurs.
 */
xmlNodePtr
xmlSecKeyValueFindNode(xmlDocPtr doc, xmlNodePtr keyValueNode, 
		       xmlSecAlgorithm alg, int addIfNotFound) {
    xmlNodePtr cur;
    const xmlChar *name = NULL;

    if(keyValueNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueFindNode: keyValueNode is null\n");	
#endif
	return(NULL);
    }

    /* find the node or create a new one */
    switch(alg) {
    case xmlSecSignDsaSha1:
	name = BAD_CAST "DSAKeyValue";
	break;
    case xmlSecSignRsaSha1:
	name = BAD_CAST "RSAKeyValue";
	break;
    case xmlSecMacHmacSha1:
	name = BAD_CAST "HMACKeyValue";
	break;
    default:
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueFindNode: unable to find key %d\n", alg);	
	return(NULL);	
    }
    
    cur = xmlSecGetNextElementNode(keyValueNode->children);
    while(cur != NULL) {	
	if(xmlSecCheckNodeName(doc, cur, name, xmlDSigNs)) {
	    break;
	}	
	cur = xmlSecGetNextElementNode(cur->next);
    }

    if(addIfNotFound && (cur == NULL)) {
	/* add new node */
	cur = xmlSecNewSibling(keyValueNode, NULL, NULL, name);
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecKeyValueFindNode: unable to create new node\n");	
#endif
	    return(NULL);
	}
    }
    return(cur);
}

/**
 * xmlSecKeyValueWrite:
 * @key:	the key
 * @doc:	the container XML document
 * @keyValueNode:	the KeyValue node (parent for key's node)
 * @writePrivateKey:	the flag that controls wherthere private
 *		key data is written to doc
 *
 * Writes key data into the doc. If KeyValue already has a node
 * for the same algorithm key then this data are overwritten.
 * Otherwise, a new KeyValue child node is created.
 *
 * Returns 0 on success or -1 if an error occurs.
 */
int
xmlSecKeyValueWrite(const xmlSecKeyPtr key, xmlDocPtr doc, 
		    xmlNodePtr keyValueNode, int writePrivateKey) {
    int ret;    


    if(key == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueWrite: key is null\n");	
#endif
	return(-1);
    }

    switch(key->algorithm) {
    case xmlSecMacHmacSha1:
	ret = xmlSecHmacKeyWrite((xmlSecHmacKeyPtr)key, doc, keyValueNode, writePrivateKey);
	break;
    case xmlSecSignDsaSha1:
	ret = xmlSecDsaKeyWrite((xmlSecDsaKeyPtr)key, doc, keyValueNode, writePrivateKey);
	break;
    case xmlSecSignRsaSha1:
	ret = xmlSecRsaKeyWrite((xmlSecRsaKeyPtr)key, doc, keyValueNode, writePrivateKey);
	break;
    default:
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueWrite: unable to write key %d\n", key->algorithm);	
	return(-1);	
    }

    if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueWrite: writing key %d failed\n", key->algorithm);	
#endif
	return(-1);
    }
    
    return(0);
}

/**
 * xmlSecKeyValueAdd:
 * @doc: 	the container XML doc
 * @parent:	the parent node
 * @prev:	the previous node.
 *
 * Adds new KeyValue node to the doc.
 *
 * Returns the pointer to new node in the doc or NLL if an error occurs. 
 */
xmlNodePtr
xmlSecKeyValueAdd(xmlDocPtr doc, xmlNodePtr parent, xmlNodePtr prev) {
    xmlNodePtr keyValueNode;
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* todo: add namespaces support */    
    /* add new KeyValue node */
    keyValueNode = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "KeyValue");
    if(keyValueNode == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyValueAdd: failed to create KeyValue node\n");	
#endif
	return(NULL);
    }
	
    return(keyValueNode);
}

/**
 * http://www.w3.org/TR/xmlSec-core/#sec-KeyInfo
 *
 * The KeyInfo Element
 *
 * KeyInfo is an optional element that enables the recipient(s) to obtain 
 * the key needed to validate the signature.  KeyInfo may contain keys, 
 * names, certificates and other public key management information, such as 
 * in-band key distribution or key agreement data. 
 * 
 *  Schema Definition:
 *
 *  <element name="KeyInfo" type="ds:KeyInfoType"/> 
 *  <complexType name="KeyInfoType" mixed="true">
 *    <choice maxOccurs="unbounded"> 
 *       <element ref="ds:KeyName"/> 
 *       <element ref="ds:KeyValue"/> 
 *       <element ref="ds:RetrievalMethod"/> 
 *       <element ref="ds:X509Data"/> 
 *       <element ref="ds:PGPData"/> 
 *       <element ref="ds:SPKIData"/>
 *       <element ref="ds:MgmtData"/>
 *       <any processContents="lax" namespace="##other"/>
 *       <!-- (1,1) elements from (0,unbounded) namespaces -->
 *    </choice>
 *    <attribute name="Id" type="ID" use="optional"/>
 *  </complexType>
 *    
 * DTD:
 *    
 * <!ELEMENT KeyInfo (#PCDATA|KeyName|KeyValue|RetrievalMethod|
 *                    X509Data|PGPData|SPKIData|MgmtData %KeyInfo.ANY;)* >      
 * <!ATTLIST KeyInfo  Id  ID   #IMPLIED >
 *  
 */ 
/**
 * xmlSecKeyInfoRead:
 * @doc:	the container XML doc
 * @keyInfoNode:	the KeyInfo node
 * @alg:	the algorithm
 *
 * Reads key from KeyInfo.
 *
 * Returns a pointer to xmlSecKey (caller is responsible for destroying it) or
 * NULL if there is no such key or an error occurs.
 */
xmlSecKeyPtr
xmlSecKeyInfoRead(const xmlDocPtr doc, const xmlNodePtr keyInfoNode, 
		  xmlSecAlgorithm alg) {
    xmlNodePtr keyNameNode = NULL;
    xmlSecKeyPtr key = NULL;
    xmlNodePtr cur;
    
    if(keyInfoNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoRead: keyInfoNode is null\n");	
#endif
	return(NULL);
    }
    
    /* add support for RetrievalMethod, X509Data, etc. */
    /* find name and value nodes */
    cur = xmlSecGetNextElementNode(keyInfoNode->children);
    while(cur != NULL) {
	if(xmlSecCheckNodeName(doc, cur, BAD_CAST "KeyName", xmlDSigNs)) {
	    keyNameNode = cur;
	} else if(xmlSecCheckNodeName(doc, cur, BAD_CAST "KeyValue", xmlDSigNs)) {
	    xmlChar* name = NULL;
	    
	    if(keyNameNode != NULL) {
		name = xmlNodeGetContent(keyNameNode);
	    }
	    key = xmlSecKeyValueRead(doc, cur, name);	    
	    /* we do not check error key == NULL here because there could 
	     * be unknown keys */
	    if(name != NULL) xmlFree(name);
	     
	    if((key != NULL) && ((alg == xmlSecAlgorithmUnknown) || 
			         (alg == key->algorithm))) {
		/* found! */
		break;
	    } 
	    xmlSecKeyDestroy(key);	    
	    keyNameNode = NULL;
	}
	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    /* todo: process X509 or PGP or whatever (validate key, etc) */
    return(key);
}

/**
 * xmlSecKeyInfoFindNode:
 * @doc:	the container XML doc
 * @keyInfoNode:	the KeyInfo node
 * @alg:	the algorithm
 * @addIfNotFound:	if this flag is set (1) and key node is not 
 *		found then a new node will be created and added to the doc
 *
 * Searches for the node for key for specified algorithm. 
 *
 * Returns the pointer to node or NULL if node is not found or 
 * an error occurs.
 */
xmlNodePtr
xmlSecKeyInfoFindNode(xmlDocPtr doc, xmlNodePtr keyInfoNode,                                      
		xmlSecAlgorithm alg, int addIfNotFound) {
    xmlNodePtr cur;
    xmlNodePtr ret = NULL;
    
    if(keyInfoNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoFindNode: key is null\n");	
#endif
	return(NULL);
    }

    cur = xmlSecGetNextElementNode(keyInfoNode->children);
    while(cur != NULL) {
	if(xmlSecCheckNodeName(doc, cur, BAD_CAST "KeyValue", xmlDSigNs)) {
	    ret = xmlSecKeyValueFindNode(doc, cur, alg, 0);
	    if(ret != NULL) {
		return(ret);
	    }
	}
	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    if(addIfNotFound) {
	/* add new node */
	cur = xmlSecKeyValueAdd(doc, keyInfoNode, NULL);
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecKeyInfoFindNode: unable to create new KeyValue node\n");	
#endif
	    return(NULL);
	}
	
	ret = xmlSecKeyValueFindNode(doc, cur, alg, 1);
	if(ret == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecKeyInfoFindNode: unable to add new value node\n");	
#endif
	    return(NULL);
	}
    }
    return(ret);
}

/**
 * xmlSecKeyInfoWrite:
 * @key:	the key
 * @doc:	the container XML doc
 * @keyInfoNode:	the KeyInfo node
 * @writePrivateKey:	the flag to control whether the 
 *		the private key data is written to the doc
 *
 * Writes key data into the doc. If KeyInfo node already has 
 * a node with data for same algorithm key then this data
 * are replaced with new ones. 
 *
 * Returns 0 for success or -1 if an error occurs.
 */
int
xmlSecKeyInfoWrite(const xmlSecKeyPtr key, xmlDocPtr doc, 
		   xmlNodePtr keyInfoNode, int writePrivateKey) {
    xmlNodePtr cur;
    int ret;

    if((key == NULL) || (keyInfoNode == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoWrite: key or keyInfoNode is null\n");	
#endif
	return(-1);
    }

    cur = xmlSecKeyInfoFindNode(doc, keyInfoNode, key->algorithm, 1);
    if(cur == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoWrite: failed to find or create node\n");	
#endif
	return(-1);
    }


    ret = xmlSecKeyValueWrite(key, doc, cur, writePrivateKey);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoWrite: failed to write value node\n");	
#endif
	return(-1);
    }
    return(0);
}

/**
 * xmlSecKeyInfoAdd:
 * @doc:	the XML doc
 * @parent:	the parent node
 * @prev:	the prev node
 *
 * Creates new KeyInfo node in the doc.
 *
 * Returns pointer to newly created node or NULL if an error occurs.
 *
 */
xmlNodePtr
xmlSecKeyInfoAdd(xmlDocPtr doc, xmlNodePtr parent, xmlNodePtr prev) {
    xmlNodePtr keyInfoNode;
    
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* add new KeyInfo node */
    keyInfoNode = xmlSecNewSibling(parent, prev, NULL, BAD_CAST "KeyInfo");
    if(keyInfoNode == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyInfoAdd: failed to create KeyInfo node\n");	
#endif
	return(NULL);
    }

    return(keyInfoNode);
}

/** 
 * XMLSec KeyMngr
 */
/**
 * xmlSecKeyMngrDestroy:
 * @ptr:	the KeyManager
 *
 * Destroys the KeyManager by calling custom callback
 *
 */
void
xmlSecKeyMngrDestroy(xmlSecKeyMngrPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyMngrDestroy: ptr is null\n");	
#endif
    	return;
    }
    if(ptr->destroyCallback != NULL) {
	ptr->destroyCallback(ptr);
    } else {
	xmlFree(ptr);
    }
}

/**
 * xmlSecKeyMngrAddKey:
 * @ptr:	the key manager
 * @key:	the key
 * @context:	the key manager/application specific data
 * 
 * Adds new key to the key manager by calling custom callback.
 *
 * Returns 0 on success or -1 if an error occurs.
 */
int
xmlSecKeyMngrAddKey(xmlSecKeyMngrPtr ptr, xmlSecKeyPtr key, void *context) {
    if((ptr == NULL) || (key == NULL) || (ptr->addKeyCallback == NULL)) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyMngrAddKey: ptr, key or add callback is null\n");	
#endif
    	return(-1);
    }
    return ptr->addKeyCallback(ptr, key, context);
}

/**
 * xmlSecKeyMngrFindKey:
 * @ptr:	the key manager
 * @alg:	the key algorithm
 * @name:	the key name
 * @context:	the  key manager/application specific data
 *
 * Searches for key with specified parameters
 *
 * Returns pointer to the key or NULL if the key is not found 
 * or an error occurs.
 */
xmlSecKeyPtr
xmlSecKeyMngrFindKey(xmlSecKeyMngrPtr ptr, xmlSecAlgorithm alg,
		     const xmlChar* name, void *context) {
    if((ptr == NULL) || (ptr->findKeyCallback == NULL)) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecKeyMngrFindKey: ptr or find callback is null\n");	
#endif
    	return(NULL);
    }
    return ptr->findKeyCallback(ptr, alg, name, context);
}


/** 
 * Simple KeyMngr
 */
/**
 * xmlSecKeyCompare:
 * @ptr1:	the key1
 * @ptr2:	the key2
 *
 * Compares two keys by algorithm 
 *
 * Returns -1, 0 or 1 if key1 algorithm <, = or > key2 algorithm.
 */
static int
xmlSecKeyCompare (xmlSecKeyPtr ptr1, xmlSecKeyPtr ptr2) {
    if(ptr1 == ptr2) return(0);
    if(ptr1 == NULL) return(-1);
    if(ptr2 == NULL) return(1);
    if(ptr1->algorithm < ptr2->algorithm) return(-1);
    if(ptr1->algorithm > ptr2->algorithm) return(1);
    return(0);
}

/**
 * xmlSecKeyListDeallocator:
 * @lk:		the list element
 *
 * The simple key manager list element deallocator
 */
static void
xmlSecKeyListDeallocator(xmlLinkPtr lk) {
    void * data = xmlLinkGetData(lk);
    if(data == NULL) {
	return;
    }        
    xmlSecKeyDestroy((xmlSecKeyPtr)data);
}

/**
 * xmlSecSimpleKeyMngrAddKey:
 * @ptr:	the simple key manager
 * @key:	the key 
 * @context:	the context (ignored)
 *
 * Simple key manager AddKey callback.
 *
 * Returns 0 on success or -1 if an error occurs.
 */
static int
xmlSecSimpleKeyMngrAddKey(xmlSecKeyMngrPtr ptr, xmlSecKeyPtr key, void *context) {
    int ret;
    
    if((ptr == NULL) || (ptr->data == NULL) || (key == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrAddKey: ptr, ptr->data or key is null\n");	
#endif
	return(-1);
    }
    ret = xmlListInsert((xmlListPtr)(ptr->data), key);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrAddKey: failed to add new key\n");	
#endif
	return(-1);
    }
    
    return(0);
}


/**
 * xmlSecSimpleKeyMngrFindKey:
 * @ptr:	the simple key manager
 * @alg:	the algorithm
 * @name:	the key name
 * @context:	the context (ignored)
 *
 * Simple Key Manager FindKey callback
 *
 * Returns the key or NULL if it is not found or an error occurs.
 */
static xmlSecKeyPtr
xmlSecSimpleKeyMngrFindKey(const xmlSecKeyMngrPtr ptr, xmlSecAlgorithm alg,
			const xmlChar* name, void *context) {
    xmlSecKey key;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrFindKey: ptr or ptr->data is null\n");	
#endif
	return(NULL);
    }

    key.algorithm = alg;
    return(xmlListSearch((xmlListPtr)(ptr->data), &key));
}

/**
 * xmlSecSimpleKeyMngrDestroy:
 * @ptr:	the simple key manager
 *
 * Destroys the key manager
 */
static void
xmlSecSimpleKeyMngrDestroy(xmlSecKeyMngrPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrDestroy: ptr is null\n");		    
#endif
	return;
    }
    
    if(ptr->data != NULL) {
	xmlListDelete((xmlListPtr)(ptr->data));
    }
    
    memset(ptr, 0, sizeof(xmlSecSimpleKeyMngr)); 
    xmlFree(ptr);    
}

/**
 * xmlSecSimpleKeyMngrCreate:
 * 
 * Creates new simple keys manager which allows to search for keys
 * by algorithm, save and load keys to/from file.
 *
 * Returns pointer to newly allocated simple keys manager (caller is
 * responsible for deleting it) or NULL if an error occurs.
 */
xmlSecSimpleKeyMngrPtr
xmlSecSimpleKeyMngrCreate() {
    xmlSecSimpleKeyMngrPtr ptr;
    
    ptr = (xmlSecSimpleKeyMngrPtr)xmlMalloc(sizeof(xmlSecSimpleKeyMngr));
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrCreate: malloc failed\n");	
#endif
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecSimpleKeyMngr)); 
    
    ptr->destroyCallback = xmlSecSimpleKeyMngrDestroy;
    ptr->addKeyCallback = xmlSecSimpleKeyMngrAddKey;
    ptr->findKeyCallback = xmlSecSimpleKeyMngrFindKey;
    
    ptr->data = xmlListCreate((xmlListDeallocator)xmlSecKeyListDeallocator,
			      (xmlListDataCompare)xmlSecKeyCompare);
    if(ptr->data == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrCreate: failed to create keys list\n");	
#endif
	xmlSecSimpleKeyMngrDestroy(ptr);
	return(NULL);
    }
    return(ptr);
}

/**
 * xmlSecSimpleKeyMngrLoad:
 * @ptr: 	the simple key manager
 * @uri:	the uri to load keys from
 *
 * Loads keys from file.
 *
 * Returns 0 on success or -1 if an error occurs.
 */
int
xmlSecSimpleKeyMngrLoad(xmlSecKeyMngrPtr ptr, const char* uri) {
    xmlDocPtr doc;
    xmlNodePtr keysNode;
    xmlNodePtr cur;
    xmlSecKeyPtr key = NULL;
    int ret;
    
    if((ptr == NULL) || (ptr->data == NULL) || (uri == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrLoadKeys: ptr->list or uri is null\n");	
#endif
	return(-1);
    }

    doc = xmlParseFile(uri);
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrLoadKeys: failed to load keys from \"%s\"\n", uri);	
#endif
	return(-1);
    }
    
    keysNode = xmlSecGetNextElementNode(doc->children);
    while(keysNode != NULL && xmlSecCheckNodeName(doc, keysNode, BAD_CAST "Keys", xmlDSigNs)) {
	cur = xmlSecGetNextElementNode(keysNode->children);
	while(cur != NULL && xmlSecCheckNodeName(doc, cur, BAD_CAST "KeyInfo", xmlDSigNs)) {	
	    key = xmlSecKeyInfoRead(doc, cur, xmlSecAlgorithmUnknown);
	    if(key != NULL) {
		/* remove mention about node */
		key->node = NULL;
		ret = xmlListInsert((xmlListPtr)(ptr->data), key);
		if(ret < 0) {	    
#ifdef DEBUG_XMLSEC
    		    xmlGenericError(xmlGenericErrorContext,
			"xmlSecSimpleKeyMngrLoadKeys: failed to add key from \"%s\"\n", uri);	
#endif
		    xmlSecKeyDestroy(key);
		    xmlFreeDoc(doc);
		    return(-1);
		}	
	    }		
	    cur = xmlSecGetNextElementNode(cur->next);
	}
	if(cur != NULL) {
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSimpleKeyMngrLoadKeys: only KeyInfo elements are permitted inside Keys element in keys file \"%s\"\n", uri);	
	    xmlFreeDoc(doc);    
	    return(-1);
	}
	keysNode = xmlSecGetNextElementNode(keysNode->next); 
    }
    if(keysNode != NULL) {
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecSimpleKeyMngrLoadKeys: only Keys elements are permitted in keys file \"%s\"\n", uri);	
	xmlFreeDoc(doc);    
	return(-1);
    }
    
    xmlFreeDoc(doc);    
    return(0);
}

/**
 * xmlSecSimpleKeyMngrSave:
 * @ptr:	the simple key manager
 * @uri:	the uri to save keys to
 * @writePrivateKey:	the flag wherthere to write out 
 *		private keys (1) or public keys only (0)
 *
 * Saves keys from key manager to a file. The file is NOT
 * encrypted or protected! Simple Key Manager and this function
 * should be used for testing purposes only.
 *
 * Returns 0 on success or -1 on error.
 */
int
xmlSecSimpleKeyMngrSave(const xmlSecKeyMngrPtr ptr, const char* uri,
			int writePrivateKey) {
    xmlDocPtr doc;
    xmlListPtr listCopy;
    xmlLinkPtr lk;
    xmlSecKeyPtr key;
    xmlNodePtr keysNode;
    xmlNodePtr cur;
    int ret;

    if((ptr == NULL) || (ptr->data == NULL) || (uri == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: list or uri is null\n");	
#endif
	return(-1);
    }

    listCopy = xmlListDup((xmlListPtr)(ptr->data));
    if(listCopy == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: failed to dup keys list\n");	
#endif
	return(-1);
    }
    
    doc = xmlNewDoc(BAD_CAST "1.0");
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: failed to create new doc\n");	
#endif
	xmlListDelete(listCopy);
	return(-1);
    }
    
    /* add Keys node */
    keysNode = xmlNewDocNode(doc, NULL, BAD_CAST "Keys", NULL);
    if(keysNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: failed to create new Keys node\n");	
#endif
	xmlFreeDoc(doc); 
	xmlListDelete(listCopy);	
	return(-1);
    }
    xmlDocSetRootElement(doc, keysNode);
    if(xmlNewNs(keysNode, xmlDSigNs, NULL) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: failed to add ns to node\n");	
#endif
	xmlFreeDoc(doc); 
	xmlListDelete(listCopy);	
	return(-1);
    }
    
    cur = NULL;
    while(!xmlListEmpty(listCopy)) {
	lk = xmlListEnd(listCopy);
	key = (xmlSecKeyPtr)xmlLinkGetData(lk);
	if(key == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSimpleKeyMngrSaveKeys: data is NULL\n");	
#endif
	    xmlFreeDoc(doc);
	    xmlListDelete(listCopy);
	    return(-1);
	}  
	
	cur = xmlSecKeyInfoAdd(doc, keysNode, cur);
	if(cur == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSimpleKeyMngrSaveKeys: failed to create new KeyInfo node\n");	
#endif
	    xmlFreeDoc(doc); 
	    xmlListDelete(listCopy);	
	    return(-1);
	}
	
	ret = xmlSecKeyInfoWrite(key, doc, cur, writePrivateKey);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSimpleKeyMngrSaveKeys: failed to write key\n");	
#endif
	    xmlFreeDoc(doc); 
	    xmlListDelete(listCopy);	
	    return(-1);
	}	   
	xmlListPopBack(listCopy);
    }

    /* now dump everything! */
    ret = xmlSaveFormatFile(uri, doc, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSimpleKeyMngrSaveKeys: failed to write file \"%s\"\n", uri);	
#endif
	xmlFreeDoc(doc); 
	xmlListDelete(listCopy);	
	return(-1);
    }	   
    
    xmlFreeDoc(doc);
    xmlListDelete(listCopy);
    return(0);
}

