/** 
 * Encryption Algorithms: SHA1
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/sha.h>

#include <libxml/xmlmemory.h>
#include <libxml/xmlerror.h> 

#include <xmlsec/algorithms.h>
#include <xmlsec/sha1.h>


/*
 * SHA1 Binary Transform
 */
/**
 * xmlSecSha1TransformRead:
 * @ptr:	the SHA1 transform
 * @buf:	the output buffer
 * @len:	the output buffer length
 *
 * This is a Read callback for SHA1 trasnform. It reads *all* data
 * from previous transform, calculates SHA1 digest and returns result
 * to the caller
 *
 * Returns the number of bytes written or -1 if an error occurs.
 */
static int
xmlSecSha1TransformRead(xmlSecBinTransformPtr ptr, unsigned char *buf, size_t len) {
    int ret;
    SHA_CTX* ctx;        
    
    if((ptr == NULL) || (buf == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformRead: ptr, ptr->data or buffer is null\n");
#endif	    
	return(-1);
    }
    
    if(len < SHA_DIGEST_LENGTH) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformRead: input len %d is not enough (%d bytes required)\n", len, SHA_DIGEST_LENGTH);
	return(-1);
    }
    
    /* if we already called Final then nothing to read more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (SHA_CTX*)ptr->data;
    /* have to read everything first */
    do {
	ret = xmlSecBinTransformRead(ptr->prev, buf, len);
        if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecSha1TransformRead: prev read failed\n");
#endif 	    
	    return(-1);
	} else if(ret > 0) {
	    SHA1_Update(ctx, buf, ret);
	}
    } while(ret > 0);
    
    /* ret == 0 i.e. there is no more data */
    SHA1_Final(buf, ctx);
    ptr->finalized = 1;    
    return(SHA_DIGEST_LENGTH);
}

/**
 * xmlSecSha1TransformWrite:
 * @ptr:	the SHA1 trasnform
 * @buf:	the input buffer
 * @len:	the input buffer length
 *
 * This is the Write callback for SHA1 transform. We simply update
 * SHA1 context with provided data.
 *
 * Returns 0 on success or -1 on error.
 */
static int
xmlSecSha1TransformWrite(xmlSecBinTransformPtr ptr, const unsigned char *buf, size_t len) {
    SHA_CTX* ctx;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformWrite: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }
    if((buf == NULL) || (len == 0)) {
	return(0);
    }

    /* if we already called Final then nothing to write more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (SHA_CTX*)ptr->data;     
    SHA1_Update(ctx, buf, len);
    return(len);
}

/**
 * xmlSecSha1TransformFlush:
 * @ptr:	the SHA1 transform
 * 
 * Finalise SHA1 digest and writes it to next transform
 *
 * Returns the number of bytes written
 */
static int
xmlSecSha1TransformFlush(xmlSecBinTransformPtr ptr) {
    unsigned char digest[SHA_DIGEST_LENGTH];
    int ret;
    SHA_CTX* ctx;
    
    if((ptr == NULL) || (ptr->data == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformFlush: ptr, ptr->data is null\n");
#endif	    
	return(-1);
    }

    /* if we already called Final then nothing to flush more! */
    if(ptr->finalized) {
	return(0);
    }
    
    ctx = (SHA_CTX*)ptr->data; 
    SHA1_Final(digest, ctx); 
        
    ret = xmlSecBinTransformWrite(ptr->next, digest, sizeof(digest));
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformFlush: prev write failed\n");
#endif 	    
	return(-1);
    }     
    
    ret = xmlSecBinTransformFlush(ptr->next);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC    
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformFlush: prev flush failed\n");
#endif 	    
	return(-1);
    } 
    
    return(0);
}
 
/**
 * xmlSecSha1TransformDestroy:
 * @ptr:	the SHA1 transform
 *
 * Destroys the the SHA1 transform
 */
static void 
xmlSecSha1TransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform) + sizeof(SHA_CTX));
    xmlFree(ptr);
}

/**
 * xmlSecSha1TransformCreate:
 *
 * Creates new SHA1 transform.
 *
 * Returns new SHA1 transform or NULL if an error occurs.
 */
xmlSecBinTransformPtr	
xmlSecSha1TransformCreate () {
    xmlSecBinTransformPtr ptr;

    /*
     * Allocate a new xmlSecSha1Transform and fill the fields.
     */
    ptr = (xmlSecBinTransformPtr) xmlMalloc(sizeof(xmlSecBinTransform) + sizeof(SHA_CTX));
    if (ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecSha1TransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlSecBinTransform) + sizeof(SHA_CTX));

    ptr->data = ((unsigned char*)ptr) + sizeof(xmlSecBinTransform);
    SHA1_Init((SHA_CTX*)ptr->data);    
    
    ptr->algorithm = xmlSecDigestSha1;
    
    ptr->destroyCallback = (xmlSecBinTransformDestroyCallback)xmlSecSha1TransformDestroy;
    ptr->readCallback = (xmlSecBinTransformReadCallback)xmlSecSha1TransformRead;
    ptr->writeCallback = (xmlSecBinTransformWriteCallback)xmlSecSha1TransformWrite;
    ptr->flushCallback = (xmlSecBinTransformFlushCallback)xmlSecSha1TransformFlush;
    
    return(ptr);    
}

