/** 		
 * XMLSec library
 *
 * Binary transforms and binary/xml transforms chain
 *
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/c14n.h>
#include <libxml/parserInternals.h>

#include <xmlsec/xmlsec.h>
#include <xmlsec/algorithms.h>
#include <xmlsec/base64.h>
#include <xmlsec/sha1.h>
#include <xmlsec/hmac.h>
#include <xmlsec/dsa.h>
#include <xmlsec/rsa.h>
#include <xmlsec/io.h>
#include <xmlsec/xpath.h>

#include <xmlsec/transforms.h>

#define XMLSEC_TRANSFORM_BUFFER_SIZE	1024

/**
 * Binary transform
 */ 
xmlSecBinTransformPtr		
xmlSecBinTransformAdd(xmlSecBinTransformPtr cur, xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformAdd: ptr is null\n");
#endif	    
	return(NULL);
    }
    if(cur != NULL) {
	ptr->prev = cur;
	ptr->next = cur->next;
	cur->next = ptr;
	if(ptr->next != NULL) ptr->next->prev = ptr;
    } else {
	ptr->next = ptr->prev = NULL;
    }
    return(ptr);
}

xmlSecBinTransformPtr
xmlSecBinTransformAddBefore(xmlSecBinTransformPtr cur, xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformAddBefore: ptr is null\n");
#endif	    
	return(NULL);
    }
    if(cur != NULL) {
	ptr->next = cur;
	ptr->prev = cur->prev;
	cur->prev = ptr;
	if(ptr->prev != NULL) ptr->prev->next = ptr;
    } else {
	ptr->next = ptr->prev = NULL;
    }
    return(ptr);
}

void
xmlSecBinTransformRemove(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformRemove: ptr is null\n");
#endif	    
	return;
    }
    if(ptr->next != NULL) {
	ptr->next->prev = ptr->prev;
    }
    if(ptr->prev != NULL) {
	ptr->prev->next = ptr->next;
    }
    ptr->next = ptr->prev = NULL;
}

void
xmlSecBinTransformDestroy(xmlSecBinTransformPtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformDestroy: ptr is null\n");
#endif	    
	return;
    }
    xmlSecBinTransformRemove(ptr);
    if(ptr->destroyCallback != NULL) {
	ptr->destroyCallback(ptr);
    } else {
	xmlFree(ptr);
    }
}

void
xmlSecBinTransformDestroyAll(xmlSecBinTransformPtr ptr) {
    while(ptr->next != NULL) {
	xmlSecBinTransformDestroy(ptr->next);
    }
    while(ptr->prev != NULL) {
	xmlSecBinTransformDestroy(ptr->prev);
    }
    xmlSecBinTransformDestroy(ptr);
}

int
xmlSecBinTransformRead(xmlSecBinTransformPtr self, unsigned char *buf, size_t size) {
    return ((self != NULL && self->readCallback != NULL) ? 
 	    self->readCallback(self, buf, size) : 0);
}

int
xmlSecBinTransformWrite(xmlSecBinTransformPtr self, const  unsigned char *buf, size_t size) {
    return ((self != NULL && self->writeCallback != NULL) ? 
    	    self->writeCallback(self, buf, size) : 0);
}

int
xmlSecBinTransformFlush(xmlSecBinTransformPtr self) {
    return ((self != NULL && self->flushCallback != NULL) ? 
	    self->flushCallback(self) : 0);
}	    


xmlOutputBufferPtr
xmlSecBinTransformCreateOutputBuffer(xmlSecBinTransformPtr ptr, 
				    xmlCharEncodingHandlerPtr encoder) {
    xmlOutputBufferPtr ret;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformCreateOutputBuffer: ptr is null\n");
#endif	    
	return(NULL);
    }

    ret = xmlOutputBufferCreateIO((xmlOutputWriteCallback)xmlSecBinTransformWrite, 
				  (xmlOutputCloseCallback)xmlSecBinTransformFlush, 
				  (void*)ptr, encoder);
    if(ret == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecBinTransformCreateOutputBuffer: failed to create output buffer\n");
#endif	    
	return(NULL);
    }
    
    return(ret);
}


/**
 * Binary/XML transforms chain
 */
static void
xmlSecTransformDestroyXmlDoc(xmlSecTransformPtr ctx) {
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformDestroyXmlDoc: ctx is null\n");
#endif	    
	return;
    }
    if(ctx->curDoc != NULL && ctx->curDoc != ctx->initDoc) {
        xmlFreeDoc(ctx->curDoc);
    }
    if(ctx->curNodeSet != NULL && ctx->curNodeSet != ctx->initNodeSet) {
        xmlXPathFreeNodeSet(ctx->curNodeSet);
    }
    ctx->curDoc = NULL;
    ctx->curNodeSet = NULL;
}

/*
 * Return 0 for url w/o xpointer or barename XPointer (#id), 
 * 1 for XPointers '#xpointer(/)' or '#xpointer(id('ID'))' and 
 * -1 in parsing error case
 */
static int
xmlSecTransformParseUri(char* uri, char** url, char** id) {
    char* sharp;
    
    if(uri == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformParseUri: uri is null\n");
#endif	    
	return(-1);
    }
    if(url != NULL) (*url) = uri;
    if(id != NULL) (*id) = NULL;

    sharp = strchr(uri, '#');
    if(sharp == NULL) {
	return(0);
    }
    
    /* terminate url */
    *(sharp++) = '\0';
        
    if(strcmp(sharp, "xpointer(/)") == 0) {
	return(1);
    }
    
    if(strncmp(sharp, "xpointer(id('", 13) != 0) {
	/* #id ? */
	if(id != NULL) (*id) = sharp;
	return(0);
    }
    
    sharp += 13;
    if(id != NULL) (*id) = sharp;
    sharp = strchr(sharp, '\'');
    if(sharp == NULL) {
	return(-1);	
    }
    /* terminate url */
    *(sharp++) = '\0';
    if(strcmp(sharp, "))") != 0) {
	return(-1);
    }
    
    return(1);
}

xmlSecTransformPtr	
xmlSecTransformCreate(xmlDocPtr doc, xmlNodeSetPtr nodeSet, const char *uri) {
    xmlSecTransformPtr ctx;
    
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformCreate: doc is null\n");
#endif	    
	return(NULL);
    }
    
    /*
     * Allocate a new xmlSecTransform and fill the fields.
     */
    ctx = (xmlSecTransformPtr) xmlMalloc(sizeof(xmlSecTransform));
    if (ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformCreate: malloc failed\n");
#endif 	    
	return(NULL);
    }
    memset(ctx, 0, sizeof(xmlSecTransform));

    ctx->curBuf = xmlBufferCreate();
    if(ctx->curBuf == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecTransformCreate: unable to create memory buffer\n");
#endif	    
	xmlSecTransformDestroy(ctx);
        return(NULL);
    }

    ctx->initDoc = doc;
    ctx->initNodeSet = nodeSet;
    ctx->initUri = uri;
    
    
    /**
     *
     * The following examples demonstrate what the URI attribute identifies and
     * how it is dereferenced:
     *
     * - URI="http://example.com/bar.xml"
     * Identifies the octets that represent the external resource 
     * 'http://example.com/bar.xml', that is probably an XML document given 
     * its file extension. 
     * - URI="http://example.com/bar.xml#chapter1"
     * Identifies the element with ID attribute value 'chapter1' of the 
     * external XML resource 'http://example.com/bar.xml', provided as an 
     * octet stream. Again, for the sake of interoperability, the element 
     * identified as 'chapter1' should be obtained using an XPath transform 
     * rather than a URI fragment (barename XPointer resolution in external 
     * resources is not REQUIRED in this specification). 
     * - URI=""
     * Identifies the node-set (minus any comment nodes) of the XML resource 
     * containing the signature 
     * - URI="#chapter1"
     * Identifies a node-set containing the element with ID attribute value 
     * 'chapter1' of the XML resource containing the signature. XML Signature 
     * (and its applications) modify this node-set to include the element plus 
     * all descendents including namespaces and attributes -- but not comments.
     *
     */
     /* todo: add full xpointers support */
    if(uri == NULL) {
	ctx->curDoc = ctx->initDoc;
	ctx->curNodeSet = ctx->initNodeSet;
    } else if(strlen(uri) == 0) {
	ctx->curDoc = ctx->initDoc; 
	ctx->curNodeSet = xmlSecCreateChildsNodeSet(ctx->curDoc, NULL, NULL, 0);
	if(ctx->curNodeSet == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
    		"xmlSecTransformCreate: unable to create node set for uri=\"\"\n");
#endif	    
	    xmlSecTransformDestroy(ctx);
	    return(NULL);
	}
    } else if(strchr(uri, '#') != NULL) { 
	char *copy = NULL;
	char *url = NULL;
	char *id = NULL;
	int ret;
	int isXpointer = 0;
	xmlNodePtr ptr;
	
	copy = (char*)xmlMalloc(sizeof(char) * (strlen(uri) + 1));
	if(copy == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
    	        "xmlSecTransformCreate: failed to copy uri=\"%s\"\n", uri);
#endif	    
	    xmlFree(copy);	    
	    xmlSecTransformDestroy(ctx);
	    return(NULL);
	}
	strcpy(copy, uri);
	
	ret = xmlSecTransformParseUri(copy, &url, &id);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
    	        "xmlSecTransformCreate: failed to parse uri=\"%s\"\n", uri);
#endif	    
	    xmlFree(copy);	    
	    xmlSecTransformDestroy(ctx);
	    return(NULL);
	} else if(ret == 1) {
	    isXpointer = 1;
	} else if(ret == 0) {
	    isXpointer = 0;
	}
	
	if((url == NULL) || (strlen(url) == 0)) {
	    ctx->curDoc = ctx->initDoc;
	} else {
	    /* load document */
	    ctx->curDoc = xmlSecLoadFile(url);
	    if(ctx->curDoc == NULL) {
#ifdef DEBUG_XMLSEC
    		xmlGenericError(xmlGenericErrorContext,
    		    "xmlSecTransformCreate: failed to load url=\"%s\"\n", url);
#endif	    
		xmlFree(copy);	    
		xmlSecTransformDestroy(ctx);
		return(NULL);
	    }
	}
	
	if(id != NULL) {
	    /* find object with ID=id and create node set */
	    ptr = xmlSecFindNode(ctx->curDoc, NULL, BAD_CAST id);
	    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
    		xmlGenericError(xmlGenericErrorContext,
    	    	    "xmlSecTransformCreate: unable to find node with id=\"%s\"\n", id);
#endif	    
		xmlFree(copy);
		xmlSecTransformDestroy(ctx);
		return(NULL);
	    }
	} else {
	    /* if Id is not spefied then it is whole document */
	    ptr = NULL;
	}

	/* create nodes set */
	ctx->curNodeSet = xmlSecCreateChildsNodeSet(ctx->curDoc, ptr, NULL, (isXpointer) ? 1 : 0);
	if(ctx->curNodeSet == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
    		"xmlSecTransformCreate: unable to create node set\n");
#endif	    
	    xmlFree(copy);
	    xmlSecTransformDestroy(ctx);
	    return(NULL);
	}	
	xmlFree(copy);
    }
    return(ctx);
}

void
xmlSecTransformDestroy(xmlSecTransformPtr ctx) {
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformDestroy: ctx is null\n");
#endif	    
	return;
    }
    xmlSecTransformDestroyXmlDoc(ctx);
    if(ctx->curBuf != NULL) {
	xmlBufferFree(ctx->curBuf);
    }
    if(ctx->first != NULL) {
	xmlSecBinTransformDestroyAll(ctx->first);
    } else if(ctx->last != NULL) {
	xmlSecBinTransformDestroyAll(ctx->last); 
    }
    xmlFree(ctx);
}

/**
 * http://www.w3.org/TR/xmldsig-core/#sec-Base-64:
 *
 * Base64 transform
 * This transform requires an octet stream for input. If an XPath node-set 
 * (or sufficiently functional alternative) is given as input, then it is 
 * converted to an octet stream by performing operations logically equivalent 
 * to 1) applying an XPath transform with expression self::text(), then 2) 
 * taking the string-value of the node-set. Thus, if an XML element is 
 * identified by a barename XPointer in the Reference URI, and its content 
 * consists solely of base64 encoded character data, then this transform 
 * automatically strips away the start and end tags of the identified element 
 * and any of its descendant elements as well as any descendant comments and 
 * processing instructions. The output of this transform is an octet stream.
 *
 */
static int
xmlSecTransformPreBase64(const xmlDocPtr doc, const xmlNodePtr parent, 
			const xmlNodeSetPtr nodeSet, xmlOutputBufferPtr output) {
    xmlNodePtr cur;
    int ret;
    if((doc == NULL) || (output == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformPreBase64: cur or output is null\n");
#endif	    
	return(-1);
    }    
    
    if(nodeSet != NULL) {
	int i;
	/* simply walk thru all TEXT nodes */	
	for(i = 0; i < nodeSet->nodeNr; ++i) {
	    cur = nodeSet->nodeTab[i];
	    if(cur->type == XML_TEXT_NODE) {
                xmlOutputBufferWriteString(output, (char*)cur->content); 
	    }
	}
    } else if((parent == NULL) || (parent->type == XML_ELEMENT_NODE)) {
	cur = (parent != NULL) ? parent->children : doc->children;
	while(cur != NULL) {
	    ret = xmlSecTransformPreBase64(doc, cur, NULL, output);
	    if(ret < 0) {
#ifdef DEBUG_XMLSEC
		xmlGenericError(xmlGenericErrorContext,
		    "xmlSecTransformPreBase64: recursion failed\n");
#endif	    
		return(-1);
	    }
	}
    } else if(parent->type == XML_TEXT_NODE) { 
        xmlOutputBufferWriteString(output, (char*)parent->content); 	
    }
    return(0);
}

static int 
xmlSecTransformLoadUriToMemory(xmlSecTransformPtr ctx, const char *uri) {
    xmlSecBinTransformPtr ptr;
    unsigned char buffer[XMLSEC_TRANSFORM_BUFFER_SIZE];
    int ret;
    
    if((ctx == NULL) || (uri == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformLoadUriToMemory: ctx or uri is null\n");
#endif	    
	return(-1);
    }
    
    /* add the uri load at the beginning */
    ptr = xmlSecUriTransformCreate(uri, 1);
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformLoadUriToMemory: failed to create uri transform for \"%s\"\n", uri);
#endif	    
	return(-1);	
    }    
    
    if(xmlSecBinTransformAddBefore(ctx->first, ptr) == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformLoadUriToMemory: unable to add uri transform\n");
#endif	    
	xmlSecBinTransformDestroy(ptr);
	return(-1);
    }
    if(ctx->last == NULL) ctx->last = ptr;
    ctx->first = ptr;
	
    /* empty the current buffer */
    xmlBufferEmpty(ctx->curBuf);
    
    do {
	ret = xmlSecBinTransformRead(ctx->last, buffer, XMLSEC_TRANSFORM_BUFFER_SIZE);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecTransformLoadUriToMemory: failed to read from binary trasnforms chain\n");
#endif	    
	    return(-1);
	} else if(ret > 0) {
	    xmlBufferAdd(ctx->curBuf, buffer, ret);
	}
    } while(ret > 0);

    /* cleanup */
    xmlSecBinTransformDestroyAll(ctx->first);
    ctx->first = ctx->last = NULL;
    return(0);
}


static int 
xmlSecTransformDumpXmlToMemory(xmlSecTransformPtr ctx, xmlDocPtr doc, xmlNodeSetPtr nodeSet) {
    xmlSecBufferTransformPtr buffer;
    xmlOutputBufferPtr output;
    int ret;
    
    if((ctx == NULL) || (doc == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformDumpXmlToMemory: ctx or doc is null\n");
#endif	    
	return(-1);
    }

    /* first of all, add the memory buffer at the end */
    buffer = xmlSecBufferTransformCreate();
    if(buffer == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformDumpXmlToMemory: failed to create memory buffer\n");
#endif	    
	return(-1);	
    }    

    if(xmlSecBinTransformAdd(ctx->last, buffer) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransforDumpXmlToMemory: unable to add memory buffer to the chain\n");
#endif	    
	xmlSecBinTransformDestroy(buffer);
	return(-1);
    }
    if(ctx->first == NULL) ctx->first = buffer;
    ctx->last = buffer;

    /* now create output buffer for c14n */
    output = xmlSecBinTransformCreateOutputBuffer(ctx->first, NULL);
    if(output == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransforDumpXmlToMemory:  failed to create output buffer\n");
#endif	    
	return(-1);
    }

    /* 
     * by default (ctx->c14n == NULL) we use inclusive c14n:
     *
     * If the data object is a node-set and the next transform requires octets, 
     * the signature application MUST attempt to convert the node-set to an octet 
     * stream using Canonical XML [XML-C14N].  
     *
     * the story is different if the first 
     * transform is base64 decode:
     *
     * http://www.w3.org/TR/xmldsig-core/#sec-Base-64
     *
     * This transform requires an octet stream for input. If an XPath node-set 
     * (or sufficiently functional alternative) is given as input, then it is 
     * converted to an octet stream by performing operations logically equivalent 
     * to 1) applying an XPath transform with expression self::text(), then 2) 
     * taking the string-value of the node-set. Thus, if an XML element is 
     * identified by a barename XPointer in the Reference URI, and its content 
     * consists solely of base64 encoded character data, then this transform 
     * automatically strips away the start and end tags of the identified element 
     * and any of its descendant elements as well as any descendant comments and 
     * processing instructions. The output of this transform is an octet stream.
     */
    if(ctx->c14n == NULL && (!ctx->first->encode) && (ctx->first->algorithm == xmlSecEncBase64)) {
        ret = xmlSecTransformPreBase64(doc, NULL, nodeSet, output);
    } else {
        ret = xmlSecC14NTransformExecute(ctx->c14n, doc, nodeSet, output);
        if(ctx->c14n != NULL) {
	    xmlSecC14NTransformDestroy(ctx->c14n);
	    ctx->c14n = NULL;
	}
    }    
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransforDumpXmlToMemory: c14n algorithm failed\n");
#endif	    
	xmlOutputBufferClose(output);
	return(-1);	
    }

    /* flush data in the buffer by closing it */    
    ret = xmlOutputBufferClose(output);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransforDumpXmlToMemory: buffer flush failed\n");
#endif	    
	return(-1);	
    }

    /* now copy results to our internal buffer */
    /* todo: probably we can "reassign" the buffer */
    xmlBufferEmpty(ctx->curBuf);
    xmlBufferAdd(ctx->curBuf, 
	    xmlBufferContent(xmlSecBufferTransformGet(buffer)),
	    xmlBufferLength(xmlSecBufferTransformGet(buffer)));
    
    /* cleanup */    
    xmlSecBinTransformDestroyAll(ctx->first);
    ctx->first = ctx->last = NULL;
    xmlSecTransformDestroyXmlDoc(ctx);

    return(0);
}

static int 
xmlSecTransformDumpToMemory(xmlSecTransformPtr ctx) {
    int ret;
    
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformDumpToMemory: ctx is null\n");
#endif	    
	return(-1);
    }

    if(ctx->curDoc != NULL) {
        ret = xmlSecTransformDumpXmlToMemory(ctx, ctx->curDoc, ctx->curNodeSet);
    } else if(ctx->initUri != NULL) {
        ret = xmlSecTransformLoadUriToMemory(ctx, (char*)ctx->initUri);	
    } else {
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecTransformDumpToMemory: both doc and uri are null\n");
	return(-1);
    }
    
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecTransformDumpToMemory: failed to dump results to memory\n");
#endif	    
        return(-1);
    }
    return(0);
}

static int
xmlSecTransformCreateXml(xmlSecTransformPtr ctx) {
    int ret;

    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformCreateXml: ctx is null\n");
#endif	    
	return(-1);
    }

    if((ctx->curDoc == NULL) && (ctx->initUri == NULL)) { 
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecTransformCreateXml: both curDoc and uri are null\n");
        return(-1);
    }

    if((ctx->curDoc == NULL) && (ctx->first == NULL)) {
	/* load XML document directly from file */
	ctx->curDoc = xmlSecLoadFile(ctx->initUri);
	if(ctx->curDoc == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
	        "xmlSecTransformCreateXml: unable to load xml file \"%s\"\n", 
	        (ctx->initUri != NULL) ? ctx->initUri : "null" );
#endif	    
	    return(-1);
	}
	ctx->curNodeSet = NULL;
    } else if((ctx->first != NULL) || (ctx->c14n != NULL)) { 
        /* 
         * bin transforms chain is defined or c14n is pending
         * the source is curDoc 
         */
        ret = xmlSecTransformDumpToMemory(ctx);
        if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
	        "xmlSecTransformCreateXml: failed to dump results to memory\n");
#endif	    
	    return(-1);
	}
	/* parse XML doc from memory */
	ctx->curDoc = xmlSecParseMemory((char*)xmlBufferContent(ctx->curBuf), 
		    			xmlBufferLength(ctx->curBuf));
	if(ctx->curDoc == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
	        "xmlSecTransformCreateXml: failed to load xml from memory\n");
#endif	    
	    return(-1);	    
	}
	/* do not forget to empty buffer! */
	xmlBufferEmpty(ctx->curBuf);	
    }
    return(0);
}

int
xmlSecTransformUpdate(xmlSecTransformPtr ctx, xmlSecAlgorithm alg, 
		      int encode, xmlNodePtr transformNode,
		      void *key, void *data) {
    xmlSecBinTransformPtr ptr = NULL;
    xmlDocPtr newDoc = NULL;
    xmlNodeSetPtr newNodeSet = NULL;
    int ret;
    
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformUpdate: ctx is null\n");
#endif	    
	return(-1);
    }
    
    switch(alg) {
    case xmlSecC14NInclusive:
    case xmlSecC14NInclusiveWithComments:
    case xmlSecC14NExclusive:
    case xmlSecC14NExclusiveWithComments:
	ret = xmlSecTransformCreateXml(ctx);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecTransformUpdate: unable to create xml document\n");
#endif	    
	    return(-1);
	}
	ctx->c14n = xmlSecC14NTransformCreate(alg, transformNode);
	return(0);
    case xmlSecTransformXslt:
	/* todo: */
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformUpdate: transform %d is not valid as transform\n", alg);
	return(-1);
    case xmlSecTransformXpath:
	ret = xmlSecTransformCreateXml(ctx);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecTransformUpdate: unable to create xml document\n");
#endif	    
	    return(-1);
	}
	newNodeSet = xmlSecXPathTransformExecute(ctx->initDoc, transformNode,
						ctx->curDoc, ctx->curNodeSet);
	break;
    case xmlSecTransformEnvSign:
	ret = xmlSecTransformCreateXml(ctx);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecTransformUpdate: unable to create xml document\n");
#endif	    
	    return(-1);
	}
	newNodeSet = xmlSecEnvSignTransformExecute(ctx->initDoc, transformNode,
						ctx->curDoc, ctx->curNodeSet); 
	break;
    case xmlSecDigestSha1:
	ptr = xmlSecSha1TransformCreate();
	break;
    case xmlSecEncBase64:
	ptr = xmlSecBase64TransformCreate(transformNode, encode);
	break;
    case xmlSecMacHmacSha1:
	ptr = xmlSecHmacTransformCreate(transformNode, (const xmlSecHmacKeyPtr)key);
	break;
    case xmlSecSignDsaSha1:
	ptr = xmlSecDsaSha1TransformCreate(encode, (const xmlSecDsaKeyPtr)key, 
					   (const xmlChar*)data);
	break;
    case xmlSecSignRsaSha1:
	ptr = xmlSecRsaSha1TransformCreate(encode, (const xmlSecRsaKeyPtr)key, 
					   (const xmlChar*)data);
	break;
    case xmlSecAlgorithmDebugDump:
	ptr = xmlSecDebugDumpTransformCreate((char*)key);
	break;
    default:
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformUpdate: transform %d is not valid as transform\n", alg);
	return(-1);
    }

    if((ptr == NULL) && (newNodeSet == NULL) && (newDoc == NULL)) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformUpdate: unable to execute transform %d\n", alg);
#endif	    
	return(-1);
    }
    if(ptr != NULL) {
    	if(xmlSecBinTransformAdd(ctx->last, ptr) == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecTransformUpdate: unable to add binary transform %d\n", alg);
#endif	    
	    xmlSecBinTransformDestroy(ptr);
	    return(-1);
	}
	if(ctx->first == NULL) ctx->first = ptr;
	ctx->last = ptr;
    } else if(newDoc != NULL) {
	xmlSecTransformDestroyXmlDoc(ctx);  
	ctx->curDoc = newDoc;
	ctx->curNodeSet = newNodeSet;
    } else {
	if((ctx->curNodeSet != NULL) && (ctx->curNodeSet != ctx->initNodeSet)) {
    	    xmlXPathFreeNodeSet(ctx->curNodeSet);
	}
	ctx->curNodeSet = newNodeSet;	
    }	
    return(0);
}

int
xmlSecTransformFinal(xmlSecTransformPtr ctx, int binary) {
    int ret;
    
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlSecTransformUpdate: ctx is null\n");
#endif	    
	return(-1);
    }
    
    if(binary) {
	ret = xmlSecTransformDumpToMemory(ctx);
    } else {
	ret = xmlSecTransformCreateXml(ctx);
    }
    
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
    	    "xmlSecTransformFinal: failed to finalize results\n");
#endif	    
	return(-1);
    }
    return(0);
}


