
/** 
 * "XML Digital Signture" implementation
 *
 *  http://www.w3.org/Signature/Overview.html
 *  "XML-Signature Syntax and Processing"
 *   http://www.w3.org/TR/xmldsig-core/
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlerror.h>
#include <libxml/globals.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <xmlsec/xmlsec.h>
#include <xmlsec/algorithms.h> 
#include <xmlsec/xmldsig.h>

#include <xmlsec/base64.h>
#include <xmlsec/hmac.h>
#include <xmlsec/dsa.h>
#include <xmlsec/xpath.h>

typedef struct _xmlDSigSignature {
    xmlDSigCtxPtr	ctx;
    xmlDocPtr		doc;
    xmlNodePtr		self;
    int			sign;
    xmlSecTransformPtr	signature;
    
    /* pointers to parts of Signature itself and its ancestors */
    xmlNodePtr		c14nMethodNode;
    xmlNodePtr		signMethodNode;
    xmlNodePtr		signedInfoNode;
    xmlNodePtr		firstRefNode;
    xmlNodePtr		lastRefNode;
    xmlNodePtr		valueNode;
    xmlNodePtr		keyInfoNode;
    xmlNodePtr		firstObjectNode;
    xmlNodePtr		lastObjectNode;	
} xmlDSigSignature, *xmlDSigSignaturePtr;

typedef struct _xmlDSigReference {
    xmlDSigCtxPtr       ctx;
    xmlDocPtr		doc;
    xmlDSigSignaturePtr	sign;
    xmlNodePtr		self;
        
    char		*uri;
    xmlSecTransformPtr	transforms;
    
    xmlNodePtr		firstTransformNode;
    xmlNodePtr		lastTransformNode;
    xmlNodePtr		methodNode;
    xmlNodePtr		valueNode;
} xmlDSigReference, *xmlDSigReferencePtr;


/*
 * Signature methods
 */
static xmlDSigSignaturePtr xmlDSigSignatureCreate 	(xmlDSigCtxPtr ctx,
							 xmlDocPtr doc,	
						         xmlNodePtr cur);
static void		xmlDSigSignatureDestroy		(xmlDSigSignaturePtr ptr);
static int		xmlDSigSignatureGenerate	(xmlDSigSignaturePtr ptr);
static int		xmlDSigSignatureValidate	(xmlDSigSignaturePtr ptr);
static int		xmlDSigSignatureMethodRead	(xmlDSigSignaturePtr ptr, 
							 xmlNodePtr self);

/*
 * Reference methods
 */
static xmlDSigReferencePtr xmlDSigReferenceCreate	(xmlDSigSignaturePtr sign,
						         xmlNodePtr cur);
static void		xmlDSigReferenceDestroy		(xmlDSigReferencePtr ptr);
static int		xmlDSigReferenceGenerate	(xmlDSigReferencePtr ptr);
static int		xmlDSigReferenceValidate	(xmlDSigReferencePtr ptr);


static int
xmlDSigTransformAddValue(xmlSecTransformPtr transform, xmlNodePtr cur) {
    if((transform == NULL) || (transform->curBuf == NULL) || (cur == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformAddValue: transform, buffer or value node is null\n");	
#endif
	return(-1);
    }
    
    xmlNodeSetContentLen(cur, xmlBufferContent(transform->curBuf), 
			      xmlBufferLength(transform->curBuf));
    return(0);
}

static int
xmlDSigCompareValue(xmlSecTransformPtr transform, xmlNodePtr cur) {
    xmlChar* content;
    int ret;
    
    if((transform == NULL) || ( transform->curBuf == NULL) || (cur == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCompareValue: transform, buffer or value node is null\n");	
#endif
	return(-1);
    }

    content = xmlNodeGetContent(cur);
    if(content == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCompareValue: node value is null\n");	
#endif
	return(-1);
    }
    
    /* decode in-place since base64 decoding take *less* space! */
    ret = xmlSecBase64Decode(content, (unsigned char*)content, xmlStrlen(content));
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCompareValue: failed to base64 decode the value\n");	
#endif
	xmlFree(content);
	return(-1);
    }
    
    if(xmlBufferLength(transform->curBuf) != ret) {
	xmlFree(content);
	return(-1);
    }
    if(memcmp(xmlBufferContent(transform->curBuf), content, ret) != 0) {
	xmlFree(content);
	return(-1);
    }

    xmlFree(content);
    return(0);
}


static int
xmlDSigSignatureKeyWrite(xmlDSigSignaturePtr ptr, xmlSecKeyPtr key) {
    int ret;

    if((ptr == NULL) || (key == NULL)) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureKeyWrite: ptr or key is null\n");	
#endif
	return(-1);
    }
    
    if(ptr->keyInfoNode == NULL) {
        ptr->keyInfoNode = xmlSecKeyInfoAdd(ptr->doc, ptr->self, ptr->valueNode);
        if(ptr->keyInfoNode == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureKeyWrite: failed to create new KeyInfo node\n");	
#endif
	    return(-1);
	}	    
    }
    
    /* do not include private key!!! */
    ret = xmlSecKeyInfoWrite(key, ptr->doc, ptr->keyInfoNode, 0);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureKeyWrite: failed to write key for transform %d\n", key->algorithm);
#endif
	return(-1);
    }	
    return(0);
}

/***********************************************************************
 *
 * The SignedInfo  Element (http://www.w3.org/TR/xmldsig-core/#sec-SignedInfo)
 * 
 * The structure of SignedInfo includes the canonicalization algorithm, 
 * a signature algorithm, and one or more references. The SignedInfo element 
 * may contain an optional ID attribute that will allow it to be referenced by 
 * other signatures and objects.
 *
 * SignedInfo does not include explicit signature or digest properties (such as
 * calculation time, cryptographic device serial number, etc.). If an 
 * application needs to associate properties with the signature or digest, 
 * it may include such information in a SignatureProperties element within 
 * an Object element.
 *
 * Schema Definition:
 *
 *  <element name="SignedInfo" type="ds:SignedInfoType"/> 
 *  <complexType name="SignedInfoType">
 *    <sequence> 
 *      <element ref="ds:CanonicalizationMethod"/>
 *      <element ref="ds:SignatureMethod"/> 
 *      <element ref="ds:Reference" maxOccurs="unbounded"/> 
 *    </sequence> 
 *    <attribute name="Id" type="ID" use="optional"/> 
 *  </complexType>
 *    
 * DTD:
 *    
 *  <!ELEMENT SignedInfo (CanonicalizationMethod, SignatureMethod,  Reference+) >
 *  <!ATTLIST SignedInfo  Id   ID      #IMPLIED>
 *  
 ***********************************************************************/
static int
xmlDSigSignedInfoRead(xmlDSigSignaturePtr ptr, xmlNodePtr self) {       
    xmlNodePtr cur;
    
    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignedInfoRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    cur = xmlSecGetNextElementNode(self->children);
    
    /* first node is CanonicalizationMethod */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "CanonicalizationMethod", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignedInfoRead: required element \"CanonicalizationMethod\" missed\n");
	return(-1);
    }
    ptr->c14nMethodNode = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next node is SignatureMethod */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "SignatureMethod", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignedInfoRead: required element \"SignatureMethod\" missed\n");
	return(-1);
    }
    ptr->signMethodNode = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is Reference nodes (at least one must present!) */
    while((cur != NULL) && xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "Reference", xmlDSigNs)) {
	if(ptr->firstRefNode == NULL) {
	    ptr->firstRefNode = cur;
	}
	ptr->lastRefNode = cur;
	cur = xmlSecGetNextElementNode(cur->next); 
    }
    if(ptr->firstRefNode == NULL) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignedInfoRead: required element \"Reference\" missed\n");
	    return(-1);
    }
    
    if(cur != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignedInfoRead: unexpected node found\n");
	return(-1);
    }
    return(0);    
}

/***********************************************************************
 *
 * The SignatureMethod Element (http://www.w3.org/TR/xmldsig-core/#sec-SignatureMethod) 
 * 
 * SignatureMethod is a required element that specifies the algorithm used 
 * for signature generation and validation. This algorithm identifies all 
 * cryptographic functions involved in the signature operation (e.g. hashing, 
 * public key algorithms, MACs, padding, etc.). This element uses the general 
 * structure here for algorithms described in section 6.1: Algorithm 
 * Identifiers and Implementation Requirements. While there is a single 
 * identifier, that identifier may specify a format containing multiple 
 * distinct signature values.
 *
 * Schema Definition:
 *
 * <element name="SignatureMethod" type="ds:SignatureMethodType"/>
 * <complexType name="SignatureMethodType" mixed="true">
 *   <sequence>
 *     <element name="HMACOutputLength" minOccurs="0" type="ds:HMACOutputLengthType"/>
 *     <any namespace="##other" minOccurs="0" maxOccurs="unbounded"/>
 *     <!-- (0,unbounded) elements from (1,1) external namespace -->
 *   </sequence>
 *   <attribute name="Algorithm" type="anyURI" use="required"/> 
 * </complexType>
 *    
 * DTD:
 *    
 * <!ELEMENT SignatureMethod (#PCDATA|HMACOutputLength %Method.ANY;)* >
 * <!ATTLIST SignatureMethod Algorithm CDATA #REQUIRED >
 *	   
 ***********************************************************************/
static int
xmlDSigSignatureMethodRead(xmlDSigSignaturePtr ptr, xmlNodePtr self) {
    xmlChar* algorithm;
    xmlSecAlgorithm alg;
    xmlSecKeyPtr key = NULL;
    xmlSecKeyPtr keyNew = NULL;
    xmlChar* signatureValue = NULL;    
    void* data = NULL;
    int ret;    

    if((ptr == NULL) || (ptr->ctx == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    
    algorithm = xmlGetProp(self, BAD_CAST "Algorithm");
    if(algorithm == NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureMethodRead: \"Algorithm\" is required property for Transform node\n");
	return(-1);
    }
    if(!xmlSecAlgCheckUsageHref(algorithm, &alg, xmlSecAlgUsageDSigSignature)) {  
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureMethodRead: Algorithm \"%s\" is not found or could not be used for signature\n", algorithm);
	xmlFree(algorithm);
	return(-1);
    }
    xmlFree(algorithm); algorithm = NULL;

    /* todo: try to read key from document */ 
    if(ptr->keyInfoNode != NULL) {
	key = keyNew = xmlSecKeyInfoRead(ptr->doc,  ptr->keyInfoNode, alg);	
    }

    if((key == NULL) || (ptr->sign && (!key->privateKey))) {
	key = xmlSecKeyMngrFindKey(ptr->ctx->keyMgr, alg, NULL, ptr->ctx->context);
    }
    if((key == NULL) || (ptr->sign && (!key->privateKey))) {
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodRead: no valid key found for alg = %d\n", alg);	
	if(keyNew != NULL) {
	    xmlSecKeyDestroy(keyNew);
	}
    	return(-1);		
    }	

    /* todo: load any additional info for algorithm */
    switch(alg) {
    case xmlSecSignRsaSha1: 
    case xmlSecSignDsaSha1:
	    /* 
	     * if there is no KeyInfo node in the template then 
	     * we will NOT write key unless it is required in the
	     * context;
	     * the signature validation application is responsible 
	     * for getting key from context
	     */
	    if(ptr->sign && ((ptr->keyInfoNode != NULL) || ptr->ctx->writeKey)){    
		ret = xmlDSigSignatureKeyWrite(ptr, key);
		if(ret < 0) {
#ifdef DEBUG_XMLSEC
    		    xmlGenericError(xmlGenericErrorContext,
			"xmlDSigSignatureMethodRead: failed to write key for alg = %d\n", alg);	
#endif
		    if(keyNew != NULL) {
			xmlSecKeyDestroy(keyNew);
		    }
		    return(-1);		
		}
	    } else if(!ptr->sign) {
		/* get signature value */
		signatureValue = xmlNodeGetContent(ptr->valueNode);
		data = signatureValue;
	    }
	    break;
    default:
	/* do nothing  */
	break;
    }
    
    ret = xmlSecTransformUpdate(ptr->signature, alg, ptr->sign, self, key, data);

    /* first destroy any alg data !*/
    if(signatureValue != NULL) {
	xmlFree(signatureValue);
    }
    if(keyNew != NULL) {
        xmlSecKeyDestroy(keyNew);
    }

    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodRead: failed to add transform %d\n", alg);
#endif
	return(-1);
    }
    return(0);
}

static xmlNodePtr
xmlDSigSignatureMethodAdd(xmlDocPtr doc, xmlNodePtr parent, xmlNodePtr prev,
			  xmlSecAlgorithm method, void *data) {
    xmlNodePtr node;
    const xmlChar* alg;
    
    if((doc == NULL) || (parent == NULL)){
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* get algorithm */
    alg = xmlSecAlgCheckUsageAlg(method, xmlSecAlgUsageDSigSignature);
    if(alg == NULL) {
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodAdd: failed to find digest alg for method %d\n", method);	
    	return(NULL);	
    }
    
    /* todo: add namespace support */
    /* add new KeyValue node */
    node = xmlSecNewSibling(parent, NULL, NULL,  BAD_CAST "SignatureMethod");
    if(node == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodAdd: failed to create node\n");	
#endif
	return(NULL);
    }
    
    if(xmlSetProp(node, BAD_CAST "Algorithm", alg) == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureMethodAdd: failed to add Algorithm=\"%s\"\n", alg);	
#endif
	xmlUnlinkNode(node);
	xmlFreeNode(node);
	return(NULL);
    } 
    
    /* todo: add HMAC data, etc. */
    return(node);
}


/***********************************************************************
 *
 * The CanonicalizationMethod Element (http://www.w3.org/TR/xmldsig-core/#sec-CanonicalizationMethod) 
 *
 * CanonicalizationMethod is a required element that specifies the 
 * canonicalization algorithm applied to the SignedInfo element prior 
 * to performing signature calculations. This element uses the general 
 * structure for algorithms described in Algorithm Identifiers and 
 * Implementation Requirements (section 6.1). Implementations MUST support 
 * the REQUIRED canonicalization algorithms.
 *
 *  Schema Definition:
 *
 *  <element name="CanonicalizationMethod" type="ds:CanonicalizationMethodType"/> 
 *  <complexType name="CanonicalizationMethodType" mixed="true">
 *    <sequence>
 *      <any namespace="##any" minOccurs="0" maxOccurs="unbounded"/>
 *      <!-- (0,unbounded) elements from (1,1) namespace -->
 *    </sequence>
 *    <attribute name="Algorithm" type="anyURI" use="required"/> 
 *  </complexType>
 *    
 *  DTD:
 *
 *  <!ELEMENT CanonicalizationMethod (#PCDATA %Method.ANY;)* > 
 *  <!ATTLIST CanonicalizationMethod Algorithm CDATA #REQUIRED >
 *
 ***********************************************************************/
static int
xmlDSigC14NMethodRead(xmlDSigSignaturePtr ptr, xmlNodePtr self) {
    xmlChar* algorithm;
    xmlSecAlgorithm alg;
    xmlNodeSetPtr nodeSet = NULL;
    int ret;    

    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    
    algorithm = xmlGetProp(self, BAD_CAST "Algorithm");
    if(algorithm == NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigC14NMethodRead: \"Algorithm\" is required property for Transform node\n");
	return(-1);
    }

    if(!xmlSecAlgCheckUsageHref(algorithm, &alg, xmlSecAlgUsageDSigC14N)) {  
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigC14NMethodRead: Algorithm \"%s\" is not found or could not be used for C14N\n", algorithm);
	xmlFree(algorithm);	
	return(-1);
    }
    xmlFree(algorithm); algorithm = NULL;

    /* 
     *  The way in which the SignedInfo element is presented to the 
     *  canonicalization method is dependent on that method. The following 
     *  applies to algorithms which process XML as nodes or characters:
     *
     *  - XML based canonicalization implementations MUST be provided with 
     *  a [XPath] node-set originally formed from the document containing 
     *  the SignedInfo and currently indicating the SignedInfo, its descendants,
     *  and the attribute and namespace nodes of SignedInfo and its descendant 
     *  elements.
     *
     *  - Text based canonicalization algorithms (such as CRLF and charset 
     *  normalization) should be provided with the UTF-8 octets that represent 
     *  the well-formed SignedInfo element, from the first character to the 
     *  last character of the XML representation, inclusive. This includes 
     *  the entire text of the start and end tags of the SignedInfo element 
     *  as well as all descendant markup and character data (i.e., the text) 
     *  between those tags. Use of text based canonicalization of SignedInfo 
     *  is NOT RECOMMENDED.   	     
     */
    switch(alg) {
    case xmlSecC14NInclusive:
    case xmlSecC14NInclusiveWithComments:
    case xmlSecC14NExclusive:
    case xmlSecC14NExclusiveWithComments:
        /* create nodes set as described above and set in the 
	 * signature transforms as current nodes set
	 */
        nodeSet = xmlSecCreateChildsNodeSet(ptr->doc, ptr->signedInfoNode, NULL, 1);
	if(nodeSet == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigC14NMethodRead: failed to create signedinfo nodes set\n");	
#endif
	    return(-1);
	}
	/* xmlSecTransform will destroy nodeSet! */
	ptr->signature->curNodeSet = nodeSet;
	ret = xmlSecTransformUpdate(ptr->signature, alg, ptr->sign, self, NULL, NULL);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigC14NMethodRead: failed to add c14n %d\n", alg);
#endif
	    return(-1);
	}
        break;
    default:
        /* other methods are not supported C14N ! */
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodRead: the algorithm %d is not allowed as c14n method\n", alg);
	return(-1);	
    }
    return(0);
}

static xmlNodePtr
xmlDSigC14NMethodAdd(xmlDocPtr doc, xmlNodePtr parent, xmlNodePtr prev,
		     xmlSecAlgorithm method, void *data) {
    xmlNodePtr node;
    const xmlChar* alg;
    
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* get algorithm */
    alg = xmlSecAlgCheckUsageAlg(method, xmlSecAlgUsageDSigC14N);
    if(alg == NULL) {
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodAdd: failed to find c14n alg for method %d\n", method);	
    	return(NULL);	
    }
    
    /* todo: add namespace support */
    /* add new KeyValue node */
    node = xmlSecNewSibling(parent, NULL, NULL, BAD_CAST "CanonicalizationMethod");
    if(node == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodAdd: failed to create node\n");	
#endif
	return(NULL);
    }
    
    if(xmlSetProp(node, BAD_CAST "Algorithm", alg) == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigC14NMethodAdd: failed to add Algorithm=\"%s\"\n", alg);	
#endif
	xmlUnlinkNode(node);
	xmlFreeNode(node);
	return(NULL);
    } 
    
    return(node);
}


/***********************************************************************
 *
 * The Signature  element (http://www.w3.org/TR/xmldsig-core/#sec-Signature)
 *
 * The Signature element is the root element of an XML Signature. 
 * Implementation MUST generate laxly schema valid [XML-schema] Signature 
 * elements as specified by the following schema:
 *
 * Schema Definition:
 *
 *  <element name="Signature" type="ds:SignatureType"/>
 *  <complexType name="SignatureType">
 *  <sequence> 
 *     <element ref="ds:SignedInfo"/> 
 *     <element ref="ds:SignatureValue"/> 
 *     <element ref="ds:KeyInfo" minOccurs="0"/> 
 *     <element ref="ds:Object" minOccurs="0" maxOccurs="unbounded"/> 
 *     </sequence> <attribute name="Id" type="ID" use="optional"/>
 *  </complexType>
 *    
 * DTD:
 *    
 *  <!ELEMENT Signature (SignedInfo, SignatureValue, KeyInfo?, Object*)  >
 *  <!ATTLIST Signature  
 *      xmlns   CDATA   #FIXED 'http://www.w3.org/2000/09/xmldsig#'
 *      Id      ID  #IMPLIED >
 *
 ***********************************************************************/
static int
xmlDSigSignatureRead(xmlDSigSignaturePtr ptr, xmlNodePtr self) {
    xmlNodePtr cur;
    int ret;
    
    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    cur = xmlSecGetNextElementNode(self->children);
    
    /* first node is SignedInfo */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "SignedInfo", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureRead: required element \"SignedInfo\" missed\n");
	    return(-1);
    }
    ptr->signedInfoNode = cur;
    ret = xmlDSigSignedInfoRead(ptr, cur);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureRead: failed to read SignedInfo node\n");	
#endif
	return(-1);
    }
    cur = xmlSecGetNextElementNode(cur->next);

    /* next node is SignatureValue */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "SignatureValue", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureRead: required element \"SignatureValue\" missed\n");
	    return(-1);
    }
    ptr->valueNode = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next node is optional KeyInfo */
    if((cur != NULL) && (xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "KeyInfo", xmlDSigNs))) {
	ptr->keyInfoNode = cur;
	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    /* next nodes are optional Object */
    while((cur != NULL) && (xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "Object", xmlDSigNs))) {
	if(ptr->firstObjectNode == NULL) {
	    ptr->firstObjectNode = cur;
	}
	ptr->lastObjectNode = cur;
	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    if(cur != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureRead: unexpected node found\n");
	return(-1);
    }
    return(0);
}

static void 
xmlDSigSignatureDestroy(xmlDSigSignaturePtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureDestroy: ptr is null\n");	
#endif
	return;
    }
    
    if(ptr->signature != NULL) {
	xmlSecTransformDestroy(ptr->signature);
    }
    
     /* 
      * Wipe memory (security!!!)
      */
     memset(ptr, 0, sizeof(xmlDSigSignature)); 
     xmlFree(ptr);
}

static int
xmlDSigAddDebugDump(xmlDSigCtxPtr ctx, xmlSecTransformPtr transforms) {
    char *filename;
    int ret;
        
    if((ctx == NULL) || (transforms == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigAddDebugDump: context or transforms is null\n");	
#endif
	return(-1);
    }
    
    if(ctx->debugFileNamesTmpl == NULL) {
	return(0);
    }
    
    filename = (char*)xmlMalloc(sizeof(xmlChar) * 
			(strlen(ctx->debugFileNamesTmpl) + 16));
    if(filename == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigAddDebugDump: failed to allocate filename\n");	
#endif
	return(-1);
    }
    
    sprintf(filename, ctx->debugFileNamesTmpl,  ctx->debugCounter++);
    ret = xmlSecTransformUpdate(transforms, xmlSecAlgorithmDebugDump, 0, NULL, filename, NULL);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigAddDebugDump: failed to add debug dump transform\n");	
#endif
	xmlFree(filename);
	return(-1);
    }
    xmlFree(filename);  
    return(0);
}

static xmlDSigSignaturePtr 
xmlDSigSignatureCreate(xmlDSigCtxPtr ctx, xmlDocPtr doc, xmlNodePtr cur) {
    xmlDSigSignaturePtr ptr;
    int ret;
    
    if((ctx == NULL) || (doc == NULL) || (cur == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCreate: context, document or node is null\n");	
#endif
	return(NULL);
    }
    
    /*
     * Allocate a new xmlDSigSignaturePtr and fill the fields.
     */
    ptr = (xmlDSigSignaturePtr) xmlMalloc(sizeof(xmlDSigSignature));
    if (ptr == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCreate: malloc failed\n");
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlDSigSignature));
    
    ptr->ctx = ctx;
    ptr->doc = doc;
    ptr->self= cur;
    
    ret = xmlDSigSignatureRead(ptr, cur);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCreate: read failed\n");	
#endif
	xmlDSigSignatureDestroy(ptr);
	return(NULL);
    }
        
    return(ptr);
}

static int
xmlDSigSignatureCalculate(xmlDSigSignaturePtr ptr, int base64) {
    int ret;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: ptr is null\n");	
#endif
	return(-1);
    }

    /* create signature transforms */    
    if(ptr->signature != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: signature is already initialized\n");
	return(-1);
    }
    ptr->signature = xmlSecTransformCreate(ptr->doc, NULL, NULL);
    if(ptr->signature == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: unable to create signature\n");	
#endif
	return(-1);
    }
    
    /* read canonicalization method and apply it */
    ret = xmlDSigC14NMethodRead(ptr, ptr->c14nMethodNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: failed to parse c14nMethod node\n");	
#endif
	return(-1);
    }

    if((ptr->ctx != NULL) && (ptr->ctx->debugFileNamesTmpl != NULL)) {
        xmlDSigAddDebugDump(ptr->ctx, ptr->signature);
    }
    
    /* read signature method and apply it */
    ret = xmlDSigSignatureMethodRead(ptr, ptr->signMethodNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: failed to parse SignatureMethod node\n");	
#endif
	return(-1);
    }

    if(base64) {	
	ret = xmlSecTransformUpdate(ptr->signature, xmlSecEncBase64, 1, NULL, NULL, NULL);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureCalculate: failed to add base64 transform\n");
#endif
	    return(-1);
        }
    }
    
    ret = xmlSecTransformFinal(ptr->signature, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureCalculate: failed to add finalize transforms\n");
#endif
	return(-1);
    }
    return(0);
}

static int
xmlDSigSignatureGenerate(xmlDSigSignaturePtr ptr) {
    int ret;
    xmlNodePtr cur;
    xmlDSigReferencePtr ref;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureGenerate: ptr is null\n");	
#endif
	return(-1);
    }
    
    if(ptr->firstRefNode == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureGenerate: no Reference nodes found\n");	
	return(-1);
    }
    
    cur = ptr->firstRefNode;
    while(cur != NULL) {
	ref = xmlDSigReferenceCreate(ptr, cur);
	if(ref == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureGenerate: ref create failed\n");	
#endif
	    return(-1);
	}

	ret = xmlDSigReferenceGenerate(ref);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureGenerate: ref create failed\n");	
#endif
	    xmlDSigReferenceDestroy(ref);	
	    return(-1);
	}

	xmlDSigReferenceDestroy(ref);	

	if(cur == ptr->lastRefNode) {
	    break;
	}
    	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    /* all reference digests were generated, now it's time 
     * to generate signature */ 
    ret = xmlDSigSignatureCalculate(ptr, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureGenerate: digest calculation failed\n");	
#endif
	return(-1);	
    }

    /* put digest in the SignatureValue */    
    ret = xmlDSigTransformAddValue(ptr->signature, ptr->valueNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureGenerate: failed to insert digest value\n");	
#endif
	return(-1);	
    }    
    return(0);
}

static int
xmlDSigSignatureValidate(xmlDSigSignaturePtr ptr) {
    int ret;
    xmlNodePtr cur;
    xmlDSigReferencePtr ref;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureValidate: ptr is null\n");	
#endif
	return(-1);
    }
    
    if(ptr->firstRefNode == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureValidate: no Reference nodes found\n");	
	return(-1);
    }
    
    cur = ptr->firstRefNode;
    while(1) {
	ref = xmlDSigReferenceCreate(ptr, cur);
	if(ref == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureValidate: ref create failed\n");	
#endif
	    return(-1);
	}

	ret = xmlDSigReferenceValidate(ref);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigSignatureValidate: ref create failed\n");	
#endif
	    xmlDSigReferenceDestroy(ref);	
	    return(-1);
	}

	xmlDSigReferenceDestroy(ref);	

	if(cur == ptr->lastRefNode) {
	    break;
	}
    	cur = xmlSecGetNextElementNode(cur->next);
    }

    /* all reference digests were validated, now it's time 
     * to validate signature */ 
    ret = xmlDSigSignatureCalculate(ptr, 0);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureValidate: digest calculation failed\n");	
#endif
	return(-1);	
    }
    /* compare with digest in the SignatureValue */    
    ret = xmlDSigCompareValue(ptr->signature, ptr->valueNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigSignatureValiate: failed to compare digest value\n");	
#endif
	return(-1);	
    }    
    return(ret);
}

/***********************************************************************
 * The Transforms Element (http://www.w3.org/TR/xmldsig-core/#sec-Transforms)
 * 
 * The optional Transforms element contains an ordered list of Transform 
 * elements; these describe how the signer obtained the data object that 
 * was digested.
 *
 * Schema Definition:
 * 
 *  <element name="Transforms" type="ds:TransformsType"/>
 *  <complexType name="TransformsType">
 *    <sequence>
 *      <element ref="ds:Transform" maxOccurs="unbounded"/> 
 *    </sequence>
 *   </complexType>
 *
 *  <element name="Transform" type="ds:TransformType"/>
 *  <complexType name="TransformType" mixed="true">
 *    <choice minOccurs="0" maxOccurs="unbounded"> 
 *      <any namespace="##other" processContents="lax"/>
 *      <!-- (1,1) elements from (0,unbounded) namespaces -->
 *      <element name="XPath" type="string"/> 
 *    </choice>
 *    <attribute name="Algorithm" type="anyURI" use="required"/> 
 *  </complexType>
 *    
 * DTD:
 *    
 *  <!ELEMENT Transforms (Transform+)>
 *  <!ELEMENT Transform (#PCDATA|XPath %Transform.ANY;)* >
 *  <!ATTLIST Transform Algorithm    CDATA    #REQUIRED >
 *  <!ELEMENT XPath (#PCDATA) >
 *	   
 ************************************************************************/

static int
xmlDSigTransformsRead(xmlDSigReferencePtr ptr, xmlNodePtr self) {
    xmlNodePtr cur;
    
    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformsRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    cur = xmlSecGetNextElementNode(self->children);
    
    /* only Transform nodes should be here */
    while((cur != NULL) && (xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "Transform", xmlDSigNs))) {
	if(ptr->firstTransformNode == NULL) {
	    ptr->firstTransformNode = cur;
	}
	ptr->lastTransformNode = cur;
	cur = xmlSecGetNextElementNode(cur->next);
    }
    
    if(cur != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigTransformsRead: unexpected node found\n");
	return(-1);
    }
    return(0);
}

static int
xmlDSigTransformRead(xmlDSigReferencePtr ptr, xmlNodePtr self, xmlSecAlgorithm* alg) {
    xmlChar* algorithm;
    
    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    
    algorithm = xmlGetProp(self, BAD_CAST "Algorithm");
    if(algorithm == NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigTransformRead: \"Algorithm\" is required property for Transform node\n");
	return(-1);
    }

    if(!xmlSecAlgCheckUsageHref(algorithm, alg, xmlSecAlgUsageDSigTransform)) {  
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigTranformRead: Algorithm \"%s\" is not found or could not be used for transform\n", algorithm);
	xmlFree(algorithm);
	return(-1);
    }
    xmlFree(algorithm); algorithm = NULL;
    
    return(0);
}

xmlNodePtr
xmlDSigTransformAdd(xmlDSigCtxPtr ctx, xmlDocPtr doc, xmlNodePtr referenceNode,
  		   xmlSecAlgorithm transformMethod, void *transformData) {
    xmlNodePtr node;
    xmlNodePtr transforms;
    const xmlChar* alg;
    int ret;
    
    if((doc == NULL) || (referenceNode == NULL)){
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* get algorithm */
    alg = xmlSecAlgCheckUsageAlg(transformMethod, xmlSecAlgUsageDSigTransform);
    if(alg == NULL) {
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformAdd: failed to find transform alg for method %d\n", transformMethod);	
    	return(NULL);	
    }

    /* find Transforms */
    transforms = xmlSecGetNextElementNode(referenceNode->children);
    if((transforms == NULL) || (!xmlSecCheckNodeName(doc, transforms, BAD_CAST "Transforms", xmlDSigNs))) {
	/* todo: add ns support and \n */
	transforms = xmlNewNode(NULL, BAD_CAST "Transforms"); 
	if(transforms == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigTransformAdd: failed to create Transforms node\n");	
#endif
	    return(NULL);	    
	}
	if(referenceNode->children == NULL) {
	    xmlAddChild(referenceNode, transforms);
	} else {
	    xmlAddPrevSibling(referenceNode->children, transforms);  
	}
    }
    
    /* todo: add namespace support */
    node = xmlSecNewSibling(transforms, NULL, NULL, BAD_CAST "Transform");
    if(node == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformAdd: failed to create node\n");	
#endif
	return(NULL);
    }
    if(xmlSetProp(node, BAD_CAST "Algorithm", alg) == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigTransformAdd: failed to add Algorithm=\"%s\"\n", alg);
#endif
	xmlUnlinkNode(node);
	xmlFreeNode(node);
	return(NULL);
    } 

    /* add data */ 
    /* todo: add namespaces support */   
    switch(transformMethod) {
    case xmlSecTransformXpath:
	ret = xmlSecXPathTrasnformAdd(doc, node, (xmlSecXPathTransformDataPtr)transformData);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigTransformAdd: failed to add XPath data\n");
#endif
	    xmlUnlinkNode(node);
	    xmlFreeNode(node);
	    return(NULL);
	} 
	break;
    case xmlSecTransformXslt:
	/* todo: */
	break;
    default:
	/* do nothing */
	break;
    }
    return(node);
}

/***********************************************************************
 *
 * The DigestMethod Element (http://www.w3.org/TR/xmldsig-core/#sec-DigestMethod)
 *
 * DigestMethod is a required element that identifies the digest algorithm 
 * to be applied to the signed object. This element uses the general structure 
 * here for algorithms specified in Algorithm Identifiers and Implementation 
 * Requirements (section 6.1).
 *
 * Schema Definition:
 *
 *   <element name="DigestMethod" type="ds:DigestMethodType"/>
 *   <complexType name="DigestMethodType" mixed="true"> 
 *     <sequence>
 *       <any namespace="##other" processContents="lax" minOccurs="0" maxOccurs="unbounded"/>
 *     </sequence> 
 *     <attribute name="Algorithm" type="anyURI" use="required"/> 
 *   </complexType>
 *    
 * DTD:
 * 
 *   <!ELEMENT DigestMethod (#PCDATA %Method.ANY;)* >
 *   <!ATTLIST DigestMethod  Algorithm       CDATA   #REQUIRED >
 *
 ***********************************************************************/ 	   
static int
xmlDSigDigestMethodRead(xmlDSigReferencePtr ptr, xmlNodePtr self) {
    xmlChar* algorithm;
    xmlSecAlgorithm alg;
    void *key = NULL;
    void *data = NULL;
    int ret;

    if((ptr == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigDigestMethogRead: ptr or self is null\n");	
#endif
	return(-1);
    }
    
    algorithm = xmlGetProp(self, BAD_CAST "Algorithm");
    if(algorithm == NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigDigestMethogRead: \"Algorithm\" is required property for Transform node\n");
	return(-1);
    }
    
    if(!xmlSecAlgCheckUsageHref(algorithm, &alg, xmlSecAlgUsageDSigDigest)) {  
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigDigestMethodRead: Algorithm \"%s\" is not found or could not be used for digest\n", algorithm);
	xmlFree(algorithm);
	return(-1);
    }
    xmlFree(algorithm); algorithm = NULL;

    /* todo: load additional data if needed */
    ret = xmlSecTransformUpdate(ptr->transforms, alg, 1, self, key, data);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigDigestMethodRead: failed to add Transform %d\n", alg);
#endif
	return(-1);
    }
    return(0);
}

/***********************************************************************
 *
 * The Reference  Element (http://www.w3.org/TR/xmldsig-core/#sec-Reference)
 * 
 * Reference is an element that may occur one or more times. It specifies 
 * a digest algorithm and digest value, and optionally an identifier of the 
 * object being signed, the type of the object, and/or a list of transforms 
 * to be applied prior to digesting. The identification (URI) and transforms 
 * describe how the digested content (i.e., the input to the digest method) 
 * was created. The Type attribute facilitates the processing of referenced 
 * data. For example, while this specification makes no requirements over 
 * external data, an application may wish to signal that the referent is a 
 * Manifest. An optional ID attribute permits a Reference to be referenced 
 * from elsewhere.
 *
 * Schema Definition:
 *
 *  <element name="Reference" type="ds:ReferenceType"/>
 *  <complexType name="ReferenceType">
 *    <sequence> 
 *      <element ref="ds:Transforms" minOccurs="0"/> 
 *      <element ref="ds:DigestMethod"/> 
 *      <element ref="ds:DigestValue"/> 
 *    </sequence>
 *    <attribute name="Id" type="ID" use="optional"/> 
 *    <attribute name="URI" type="anyURI" use="optional"/> 
 *    <attribute name="Type" type="anyURI" use="optional"/> 
 *  </complexType>
 *    
 * DTD:
 *    
 *   <!ELEMENT Reference (Transforms?, DigestMethod, DigestValue)  >
 *   <!ATTLIST Reference Id  ID  #IMPLIED
 *   		URI CDATA   #IMPLIED
 * 		Type    CDATA   #IMPLIED>
 *
 ***********************************************************************/

static int
xmlDSigReferenceRead(xmlDSigReferencePtr ptr, xmlNodePtr self) {
    xmlNodePtr cur;
    int ret;
        
    if((ptr == NULL) || (ptr->sign == NULL) || (self == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: ptr, signature  or self is null\n");	
#endif
	return(-1);
    }

    if(ptr->transforms != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: transforms are already initialized\n");
	return(-1);
    }
    
    ptr->uri = (char*)xmlGetProp(self, BAD_CAST "URI");
    /* todo: 
     * really URI is not required for *all* Reference nodes!
     *
     * http://www.w3.org/TR/xmldsig-core/#sec-URI
     * If the URI attribute is omitted altogether, the receiving application 
     * is expected to know the identity of the object. For example, a 
     * lightweight data protocol might omit this attribute given the identity 
     * of the object is part of the application context. This attribute may 
     * be omitted from at most one Reference in any particular SignedInfo, or 
     * Manifest.
     */
    if(ptr->uri == NULL) {
	 xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: URI is not specified\n");
	return(-1);
    }

    
    ptr->transforms = xmlSecTransformCreate(ptr->doc, NULL, ptr->uri);
    if(ptr->transforms == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: unable to create transforms\n");	
#endif
	return(-1);
    }
    
    cur = xmlSecGetNextElementNode(self->children);
    
    /* first node is optional Transforms */
    if((cur != NULL) && (xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "Transforms", xmlDSigNs))) {
	ret = xmlDSigTransformsRead(ptr, cur);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceRead: failed to read Transforms node\n");	
#endif
	    return(-1);
	}
	cur = xmlSecGetNextElementNode(cur->next);
    }

    /* next is required DigestMethod */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "DigestMethod", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: required element \"DigestMethod\" missed\n");
	    return(-1);
    }
    ptr->methodNode = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    /* next is required DigestValue */
    if((cur == NULL) || (!xmlSecCheckNodeName(ptr->doc, cur, BAD_CAST "DigestValue", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceRead: required element \"DigestValue\" missed\n");
	    return(-1);
    }
    ptr->valueNode = cur;
    cur = xmlSecGetNextElementNode(cur->next);

    if(cur != NULL) {
	 xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceRead: unexpected node found\n");
	return(-1);
    }
    return(0);
}

xmlNodePtr
xmlDSigReferenceAdd(xmlDSigCtxPtr ctx, xmlDocPtr doc, xmlNodePtr signatureNode,
		    const xmlChar *uri, xmlSecAlgorithm digestMethod, void *digestData) {
    xmlNodePtr node;
    xmlNodePtr tmp;
    const xmlChar* alg;
    
    if((doc == NULL) || (signatureNode == NULL)){
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceAdd: doc is null\n");	
#endif
    	return(NULL);
    }
    
    /* get algorithm */
    alg = xmlSecAlgCheckUsageAlg(digestMethod, xmlSecAlgUsageDSigDigest);
    if(alg == NULL) {
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceAdd: failed to find digest alg for method %d\n", digestMethod);	
    	return(NULL);	
    }

    /* first node is SignedInfo */
    tmp = xmlSecGetNextElementNode(signatureNode->children);
    if((tmp == NULL) || (!xmlSecCheckNodeName(doc, tmp, BAD_CAST "SignedInfo", xmlDSigNs))) {
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: first node is not \"SignedInfo\"\n");
	return(NULL);
    }
    
    /* todo: add namespace support */
    node = xmlSecNewSibling(tmp, NULL, NULL, BAD_CAST "Reference");
    if(node == NULL) {		
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceAdd: failed to create node\n");	
#endif
	return(NULL);
    }
    if(uri != NULL) {    
	if(xmlSetProp(node, BAD_CAST "URI", uri) == NULL) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceAdd: failed to add uri=\"%s\"\n", uri);
#endif
	    xmlUnlinkNode(node);
	    xmlFreeNode(node);
	    return(NULL);
	}
    } 
    
    /* DigestMethod */
    tmp = xmlSecNewSibling(node, NULL, NULL, BAD_CAST "DigestMethod");  
    if(tmp == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to create new DigestMethod node\n");
#endif
	xmlUnlinkNode(node); 
	xmlFreeNode(node);
	return(NULL);    	
    }    
    if(xmlSetProp(tmp, BAD_CAST "Algorithm", alg) == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceAdd: failed to add Algorithm=\"%s\" to DigestMethod node\n", alg);	
#endif
	xmlUnlinkNode(node); 
        xmlFreeNode(node);
	return(NULL);
    } 

    /* DigestValue */
    tmp =  xmlSecNewSibling(node, NULL, NULL, BAD_CAST "DigestValue");
    if(tmp == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to create new DigestMethod node\n");
#endif
	xmlUnlinkNode(node); 
        xmlFreeNode(node);
	return(NULL);    	
    }
    
    return(node);
}

static void 
xmlDSigReferenceDestroy(xmlDSigReferencePtr ptr) {
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceDestroy: ptr is null\n");	
#endif
	return;
    }
    
    if(ptr->transforms != NULL) {
	xmlSecTransformDestroy(ptr->transforms);
    }
    if(ptr->uri != NULL) {
	xmlFree(ptr->uri);
    }
        
     /* 
      * Wipe memory (security!!!)
      */
     memset(ptr, 0, sizeof(xmlDSigReference)); 
     xmlFree(ptr);
}

static xmlDSigReferencePtr 
xmlDSigReferenceCreate(xmlDSigSignaturePtr sign, xmlNodePtr cur) {
    xmlDSigReferencePtr ptr;
    int ret;
    
    if((sign == NULL) || (cur == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCreate: signature or node is null\n");	
#endif
	return(NULL);
    }
    
    /*
     * Allocate a new xmlDSigReferencePtr and fill the fields.
     */
    ptr = (xmlDSigReferencePtr) xmlMalloc(sizeof(xmlDSigReference));
    if (ptr == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCreate: malloc failed\n");
	return(NULL);
    }
    memset(ptr, 0, sizeof(xmlDSigReference));
    
    ptr->ctx = sign->ctx;
    ptr->doc = sign->doc;
    ptr->sign= sign;
    ptr->self= cur;
    
    ret = xmlDSigReferenceRead(ptr, cur);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCreate: read failed\n");	
#endif
	xmlDSigReferenceDestroy(ptr);
	return(NULL);
    }
        
    return(ptr);
}

static int
xmlDSigReferenceCalculate(xmlDSigReferencePtr ptr, int base64) {
    int ret;
    xmlNodePtr cur;
    xmlSecAlgorithm alg;
    int encode = 1;
    void *key = NULL;
    void *data = NULL;    
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCalculate: ptr is null\n");	
#endif
	return(-1);
    }

    cur = ptr->firstTransformNode;
    while(cur != NULL) {
	ret = xmlDSigTransformRead(ptr, cur, &alg);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceCalculate: failed to parse Transform node\n");	
#endif
	    return(-1);
	}
	
	switch(alg) {
	case xmlSecEncBase64:
	    encode = 0;
	    break;
	case xmlSecTransformXpath:
	case xmlSecTransformEnvSign:
	case xmlSecTransformXslt:
	    break;
	default:
	    /* do nothing */
	    break;
	}

	ret = xmlSecTransformUpdate(ptr->transforms, alg, encode, cur, key, data);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceCalculate: failed to add Transform %d\n", alg);
#endif
	    return(-1);
	}

	if(cur == ptr->lastTransformNode) {
	    break;
	}
	cur = xmlSecGetNextElementNode(cur->next);
    }

    if((ptr->ctx != NULL) && (ptr->ctx->debugFileNamesTmpl != NULL)) {
        xmlDSigAddDebugDump(ptr->ctx, ptr->transforms);
    }
    
    /* apply digest method */
    ret = xmlDSigDigestMethodRead(ptr, ptr->methodNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
    	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCalculate: failed to parse DigestMethod node\n");	
#endif
	return(-1);
    }

    
    if(base64) {	
	ret = xmlSecTransformUpdate(ptr->transforms, xmlSecEncBase64, 1, NULL, NULL, NULL);
	if(ret < 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigReferenceCalculate: failed to add base64 transform\n");
#endif
	    return(-1);
        }
    }
    
    ret = xmlSecTransformFinal(ptr->transforms, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceCalculate: failed to add finalize transforms\n");
#endif
	return(-1);
    }
    return(0);
}

static int
xmlDSigReferenceGenerate(xmlDSigReferencePtr ptr) {
    int ret;
    
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceGenerate: ptr is null\n");	
#endif
	return(-1);
    }

    ret = xmlDSigReferenceCalculate(ptr, 1);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceGenerate: digest calculation failed\n");	
#endif
	return(-1);	
    }

    /* put digest in the DigestValue */    
    ret = xmlDSigTransformAddValue(ptr->transforms, ptr->valueNode);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceGenerate: failed to insert digest value\n");	
#endif
	return(-1);	
    }    
    return(0);
}

static int
xmlDSigReferenceValidate(xmlDSigReferencePtr ptr) {
    int ret;
            
    if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceValidate: ptr is null\n");	
#endif
	return(-1);
    }

    ret = xmlDSigReferenceCalculate(ptr, 0);
    if(ret < 0) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceValidate: digest calculation failed\n");	
#endif
	return(-1);	
    }

    /* base64 decode the DigestValue and compare with calculated digest */    
    ret = xmlDSigCompareValue(ptr->transforms, ptr->valueNode);
    if(ret < 0) {    
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigReferenceValidate: digestvalue validation failed\n");	
#endif
	return(-1);	
    }
    return(ret);
}

/*
 * Signature Context
 */
int 
xmlDSigGenerate(xmlDSigCtxPtr ctx, xmlDocPtr doc, xmlNodePtr cur, xmlDSigResultsPtr results) {
    int ret;
    
    if((ctx == NULL) || (doc == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigGenerate: context or document is null\n");	
#endif
	return(-1);
    }
    
    /* 
     * check is the current node is a signature
     */
    if((cur != NULL) && xmlSecCheckNodeName(doc, cur, BAD_CAST "Signature", xmlDSigNs)) {
	xmlDSigSignaturePtr sign;
	    
	sign = xmlDSigSignatureCreate(ctx, doc, cur);
	if(sign == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
	        "xmlDSigGenerate: current node is Signature but creation failed\n");	
#endif	
	    return(-1);
	}
	sign->sign = 1;

	ret = xmlDSigSignatureGenerate(sign);
	if(ret < 0) { 
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigGenerate: failed to generate signature\n");
#endif	
	    xmlDSigSignatureDestroy(sign);
	    return(-1);	
	}
	xmlDSigSignatureDestroy(sign);
	return(1);
    }	

    /*  
     * walk thru childrens and calculate how many Signatures we did
     */    
    cur = (cur != NULL) ?  cur->children : doc->children;
    while(cur != NULL) {
	if(cur->type == XML_ELEMENT_NODE) {
	    ret = xmlDSigGenerate(ctx, doc, cur, results);
	    if(ret < 0) {
#ifdef DEBUG_XMLSEC
		xmlGenericError(xmlGenericErrorContext,
		    "xmlDSigGenerate: childrens check failed\n");
#endif	
		return(-1);	
	    }
	}
	cur = cur->next;
    }    
    return(0);
}

int 
xmlDSigValidate(xmlDSigCtxPtr ctx, xmlDocPtr doc, xmlNodePtr cur, xmlDSigResultsPtr results) {
    int ret;
    
    if((ctx == NULL) || (doc == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigValidate: context or document is null\n");	
#endif
	return(-1);
    }
    
    /* 
     * check is the current node is a signature
     */
    if(cur != NULL && xmlSecCheckNodeName(doc, cur, BAD_CAST "Signature", xmlDSigNs)) {
	xmlDSigSignaturePtr sign;
	    
	sign = xmlDSigSignatureCreate(ctx, doc, cur);
	if(sign == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
	        "xmlDSigValidate: current node is Signature but creation failed\n");	
#endif	
	    return(-1);
	}
	sign->sign = 0;

	ret = xmlDSigSignatureValidate(sign);
	if(ret < 0) { 
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext,
		"xmlDSigValidate: failed to generate signature\n");
#endif	
	    xmlDSigSignatureDestroy(sign);
	    return(-1);	
	}
	xmlDSigSignatureDestroy(sign);
	return(ret);
    }	

    /*  
     * walk thru childrens and calculate how many Signatures we did
     */    
    cur = (cur != NULL) ?  cur->children : doc->children;
    while(cur != NULL) {
	if(cur->type == XML_ELEMENT_NODE) {
	    ret = xmlDSigValidate(ctx, doc, cur, results);
	    if(ret < 0) {
#ifdef DEBUG_XMLSEC
		xmlGenericError(xmlGenericErrorContext,
		    "xmlDSigValidate: childrens check failed\n");
#endif	
	    	return(-1);	
	    } 
	}
	cur = cur->next;
    }    
    return(0);
}

void 
xmlDSigCtxDestroy(xmlDSigCtxPtr ctx) {
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCtxDestroy: context is null\n");	
#endif
	return;
    }

     memset(ctx, 0, sizeof(xmlDSigCtx)); 
     xmlFree(ctx);
}

xmlDSigCtxPtr 
xmlDSigCtxCreate(xmlSecKeyMngrPtr keyMgr, void *context) {
    xmlDSigCtxPtr ctx;
    
    /*
     * Allocate a new xmlDSigCtxPtr and fill the fields.
     */
    ctx = (xmlDSigCtxPtr) xmlMalloc(sizeof(xmlDSigCtx));
    if (ctx == NULL) {
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCtxCreate: malloc failed\n");
	return(NULL);
    }
    memset(ctx, 0, sizeof(xmlDSigCtx));
    
    ctx->keyMgr = keyMgr;
    ctx->context = context;
    ctx->writeKey = 1; /* be default we will include public key */
    return(ctx);
}


xmlNodePtr
xmlDSigCreate(xmlDSigCtxPtr ctx, xmlDocPtr doc,
	      xmlSecAlgorithm c14nMethod, void *c14nData,
	      xmlSecAlgorithm signMethod, void *signData) {
    xmlNodePtr signNode;
    xmlNodePtr signedInfoNode;
    xmlNodePtr signValueNode;
    
    if((ctx == NULL) || (doc == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: context or doc is null\n");	
#endif
	return(NULL);
    }
    
    signNode = xmlNewDocNode(doc, NULL, BAD_CAST "Signature", NULL);
    if(signNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to create new Signature node\n");
#endif
	return(NULL);	
    }
    
    /* add dsig namespace */
    if(xmlNewNs(signNode, xmlDSigNs, NULL) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to add DSig namespace\n");
#endif
	xmlFreeNode(signNode);
	return(NULL);    
    }

    signedInfoNode = xmlSecNewSibling(signNode, NULL, NULL, BAD_CAST "SignedInfo"); 
    if(signedInfoNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to create new SignedInfo node\n");
#endif
	xmlFreeNode(signNode);
	return(NULL);    	
    }
    
    /* create signed info node: add c14n, signature and digest value nodes */
    if(xmlDSigC14NMethodAdd(doc, signedInfoNode, NULL, c14nMethod, c14nData) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to add C14NMethod node\n");
#endif
	xmlFreeNode(signNode);
	return(NULL);    	
    }

    if(xmlDSigSignatureMethodAdd(doc, signedInfoNode, NULL, signMethod, signData) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to add SignatureMethod node\n");
#endif
	xmlFreeNode(signNode);
	return(NULL);    	
    }
    
    signValueNode = xmlSecNewSibling(signNode, NULL, NULL, BAD_CAST "SignatureValue");
    if(signValueNode == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlDSigCreate: failed to add SignatureValue node\n");
#endif
	xmlFreeNode(signNode);
	return(NULL);    	
    }

    /* todo: add data */    
    return(signNode);
}


