/** 
 * XMLSec library
 *
 * Common utility functions
 *
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/xpathInternals.h>
#include <libxml/parserInternals.h>

#include <xmlsec/xmlsec.h>


const xmlChar xmlDSigNs[] = "http://www.w3.org/2000/09/xmldsig#";

/**
 * xmlSecNewSibling:
 * @parent:	the parent node
 * @cur:	the sibling node
 * @ns:		the namespace
 * @name:	the name
 *
 * Adds new sibling node after @cur. If it is NULL then a new child
 * is added to parent.
 *
 * Returns the pointer to new node in the XML doc or NULL if an error occurs.
 */
xmlNodePtr
xmlSecNewSibling(xmlNodePtr parent, xmlNodePtr cur, xmlNsPtr ns, const xmlChar* name) {
    xmlNodePtr tmp;
    
    tmp = xmlNewNode(ns, name);
    if(tmp == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecNewSibling: failed to create node\n");	
#endif
	return(NULL);
    }
    if(xmlSecAddSibling(parent, cur, tmp) == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecNewSibling: failed to add node\n");	
#endif
	xmlFreeNode(tmp);
	return(NULL);	
    }
    return(tmp);
}

xmlNodePtr
xmlSecAddSibling(xmlNodePtr parent, xmlNodePtr prev, xmlNodePtr cur) {
    xmlNodePtr text;
    
    text = xmlNewText(BAD_CAST "\n"); 
    if(text == NULL) {	
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecAddSibling: failed to create line break text node\n");	
#endif
	return(NULL);
    }
    
    if(prev != NULL) {
	prev = xmlAddNextSibling(prev, text);
    } else if(parent != NULL) {
	prev = xmlAddChild(parent, text);
    }
    if(prev == NULL) {	
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecAddSibling: failed to add text node\n");	
#endif
	xmlFreeNode(text);
	return(NULL);
    }
    
    if(cur != NULL) {
        prev = xmlAddNextSibling(prev, cur);
    }
    return(prev);
}

xmlNodePtr
xmlSecGetNextElementNode(xmlNodePtr cur) {
    
    while((cur != NULL) && (cur->type != XML_ELEMENT_NODE)) {
	cur = cur->next;
    }
    return(cur);
}

int
xmlSecCheckNodeName(const xmlDocPtr doc, const xmlNodePtr cur, 
		     const xmlChar *name, const xmlChar *ns) {
    if(cur == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecCheckNodeName: current node is null\n");	
#endif
	return(0);
    }
    
    if(xmlStrEqual(cur->name, name)) {
	if(cur->ns == NULL && ns == NULL) {
	    return(1);
	} else if(cur->ns == NULL) {
	    xmlNsPtr tmp;

	    tmp = xmlSearchNs(doc, cur, NULL);
	    if(tmp != NULL && xmlStrEqual(tmp->href, ns)) {
		return(1);
	    }
	} else if(xmlStrEqual(cur->ns->href, ns)){
	    return(1);	
	}
    } 
    return(0);   
}

xmlNodeSetPtr
xmlSecCreateChildsNodeSet(const xmlDocPtr doc, const xmlNodePtr parent, 
			  xmlNodeSetPtr ptr, int with_comments) {
    xmlNodePtr cur;
    xmlAttrPtr attr;
    
    if(doc == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecDSigCreateChildsNodeSet: doc  is null\n");	
#endif
	if(ptr != NULL) {
	    xmlXPathFreeNodeSet(ptr);
	}
	return(NULL);	
    }
        
    /* add the node */
    if(ptr == NULL) {	
	ptr = xmlXPathNodeSetCreate(NULL);
	if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDSigCreateChildsNodeSet: failed to create node set\n");	
#endif
	    return(NULL);
	}	
    } 

    if(parent != NULL) {
        switch(parent->type) {
	case XML_COMMENT_NODE:
	    if(!with_comments) return(ptr);
	    xmlXPathNodeSetAddUnique(ptr, parent);
	    return(ptr);	
	case XML_ELEMENT_NODE:
	    xmlXPathNodeSetAddUnique(ptr, parent);
	    break;
	case XML_TEXT_NODE:
	    xmlXPathNodeSetAddUnique(ptr, parent);
	    return(ptr);
	case XML_PI_NODE:
	    xmlXPathNodeSetAddUnique(ptr, parent);
	    return(ptr);
	default:
	    return(ptr);
	}
	/* add all attrs */
	attr = parent->properties; 
        while (attr != NULL) {
	    xmlXPathNodeSetAddUnique(ptr, (xmlNodePtr)attr); 
	    attr = attr->next; 
	}	
	cur = parent->children;
    } else {
	cur = doc->children;
    }    
    
    /* add all childrens */
    while(cur != NULL) {
	ptr = xmlSecCreateChildsNodeSet(doc, cur, ptr, with_comments);
	if(ptr == NULL) {
#ifdef DEBUG_XMLSEC
    	    xmlGenericError(xmlGenericErrorContext,
		"xmlSecDSigCreateChildsNodeSet: failed to add child namespace\n");	
#endif
	    return(NULL);	    
	}
	cur = cur->next;
    }
    
    return(ptr);
}


/* 
 * Copied from xmlSAXParseFileWithData() function
 */
xmlDocPtr
xmlSecLoadFile(const char *filename) {
    xmlDocPtr ret;
    xmlParserCtxtPtr ctxt;
    char *directory = NULL;

    xmlInitParser();

    ctxt = xmlCreateFileParserCtxt(filename);
    if (ctxt == NULL) {
	return(NULL);
    }

    /* todo: set directories from current doc? */    
    if ((ctxt->directory == NULL) && (directory == NULL))
        directory = xmlParserGetDirectory(filename);
    if ((ctxt->directory == NULL) && (directory != NULL))
        ctxt->directory = (char *) xmlStrdup((xmlChar *) directory);

    /* required for c14n! */
    ctxt->loadsubset = XML_DETECT_IDS | XML_COMPLETE_ATTRS; 
    ctxt->replaceEntities = 1;
    
    xmlParseDocument(ctxt);

    if(ctxt->wellFormed) { 
	ret = ctxt->myDoc;
    } else {
       ret = NULL;
       xmlFreeDoc(ctxt->myDoc);
       ctxt->myDoc = NULL;
    }
    xmlFreeParserCtxt(ctxt);    
    return(ret);
}

/* copied from xmlSAXParseMemory */
xmlDocPtr
xmlSecParseMemory(const char *buffer, int size) {
    xmlDocPtr ret;
    xmlParserCtxtPtr ctxt;

    ctxt = xmlCreateMemoryParserCtxt(buffer, size);
    if (ctxt == NULL) return(NULL);

    /* required for c14n! */
    ctxt->loadsubset = XML_DETECT_IDS | XML_COMPLETE_ATTRS; 
    ctxt->replaceEntities = 1;

    xmlParseDocument(ctxt);

    if(ctxt->wellFormed)  {
	ret = ctxt->myDoc; 
    } else {
       ret = NULL;
       xmlFreeDoc(ctxt->myDoc);
       ctxt->myDoc = NULL;
    }
    xmlFreeParserCtxt(ctxt);    
    return(ret);
}

xmlNodePtr
xmlSecFindNode(const xmlDocPtr doc, xmlNodePtr cur, const xmlChar* id) {
    xmlNodePtr ret;
        
    if((doc == NULL) || (id == NULL)) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecFindNode: doc or id is null\n");	
#endif
	return(NULL);	
    }
    
    if((cur != NULL) && (cur->type == XML_ELEMENT_NODE)) {
	xmlChar* attr;

	attr = xmlGetProp(cur, BAD_CAST "Id");
	if(xmlStrEqual(id, attr)) {
	    xmlFree(attr);
	    return(cur);
	}
	xmlFree(attr);
    }
    
    /* add all childrens */
    cur = (cur != NULL) ? cur->children : doc->children;
    while(cur != NULL) {	
	if(cur->type == XML_ELEMENT_NODE) {
	    ret = xmlSecFindNode(doc, cur, id);
	    if(ret != NULL) {
		return(ret);	    
	    }
	}
	cur = cur->next;
    }    
    return(NULL);
}

