/** 
 * XMLSec library
 *
 * XPath and Enveloped Signature transforms
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <libxml/globals.h>
#include <libxml/xmlerror.h> 
#include <libxml/xpathInternals.h>

#include <xmlsec/xmlsec.h>
#include <xmlsec/xpath.h>


static const xmlChar xpathPattern[] = "(//. | //@* | //namespace::*)[%s]";

/** 
 * see xmlXPtrHereFunction() in xpointer.c. the only change is that 
 * we return NodeSet instead of NodeInterval
 */
static void 
xmlSecXPathHereFunction(xmlXPathParserContextPtr ctxt, int nargs) {
    CHECK_ARITY(0);

    if (ctxt->context->here == NULL)
	XP_ERROR(XPTR_SYNTAX_ERROR);
    
    valuePush(ctxt, xmlXPathNewNodeSet(ctxt->context->here));
}

static xmlXPathObjectPtr
xmlSecXPathTransformRead(xmlDocPtr doc, xmlNodePtr xpathNode, xmlNodePtr ctxNode) {
    xmlChar *expr, *tmp;
    xmlXPathObjectPtr xpath; 
    xmlXPathContextPtr ctx; 
    xmlNsPtr ns;
        
    if((doc == NULL) || (xpathNode == NULL)) {   
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformRead: doc or xpathNode is null\n");	
#endif
	return(NULL);
    }

    tmp = xmlNodeGetContent(xpathNode);
    if(tmp == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformRead: failed to get xpath node content\n");	
#endif
	return(NULL);
    }
    
    expr = (xmlChar*) xmlMalloc(sizeof(xmlChar) * (xmlStrlen(tmp) + xmlStrlen(xpathPattern) + 1));
    if(expr == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformRead: failed to allocate xpath expr buffer\n");	
#endif
	xmlFree(tmp); 
	return(NULL);
    }
    sprintf((char*) expr, (char*) xpathPattern, tmp);
    xmlFree(tmp);
    
    ctx = xmlXPathNewContext(doc);
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformRead: xpath context is null\n");	
#endif
	xmlFree(expr); 
	return(NULL);
    }

    xmlXPathRegisterFunc(ctx, (xmlChar *)"here", xmlSecXPathHereFunction);
    ctx->here = ctxNode;
    ctx->xptr = 1;
    
    /*
     * Register namespaces
     */
    ns = xpathNode->nsDef;
    while(ns != NULL) {
	if(xmlXPathRegisterNs(ctx, ns->prefix, ns->href) != 0) {
#ifdef DEBUG_XMLSEC
	    xmlGenericError(xmlGenericErrorContext, 
		"xmlSecXPathTransformRead: unable to register NS with prefix=\"%s\" and href=\"%s\"\n", ns->prefix, ns->href);
#endif
    	    xmlFree(expr); 
	    xmlXPathFreeContext(ctx); 	     
	    return(NULL);
	}
	ns = ns->next;
    }

    /*  
     * Evaluate xpath
     */
    xpath = xmlXPathEvalExpression(expr, ctx);
    if(xpath == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext, 
	    "xmlSecXPathTransformRead: xpath eval failed\n");
#endif
    	xmlFree(expr); 
	xmlXPathFreeContext(ctx); 
        return(NULL);
    }

    xmlFree(expr); 
    xmlXPathFreeContext(ctx);      
    return(xpath);
}

int
xmlSecXPathTrasnformAdd(xmlDocPtr doc, xmlNodePtr parent, 
		        xmlSecXPathTransformDataPtr data) {
    xmlNodePtr node;
    xmlNsPtr nsPtr;
    xmlChar *name;
    xmlChar *href;
    int defaultNsReplaced = 0;
            
    if((doc == NULL) || (parent == NULL)) {	
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformAdd: doc or parent is null\n");	
#endif
	return(-1);
    }
    
    node = xmlSecNewSibling(parent, NULL, NULL, BAD_CAST "XPath");
    if(node == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformAdd: failed to add XPath node\n");	
#endif
	return(-1);
    }
    
    if((data != NULL) && (data->expression != NULL)){ 
	xmlNodeSetContent(node, BAD_CAST data->expression);
    }
    if((data != NULL) && (data->namespaces != NULL)) {
	char** p;
	
	p = data->namespaces;
	while(((*p) != NULL) && ((*(p + 1)) != NULL)) {
	    href = (xmlChar*)(*(p + 1));
	    /* special name for default namespace */
	    if((strlen((*p)) == 0) || (strcmp((*p), "#default")) == 0) {
		name = NULL;
		if(!xmlStrEqual(href, xmlDSigNs)) {
		    defaultNsReplaced = 1;
		}
	    } else {
		name = (xmlChar*)(*p);
	    }
	    nsPtr = xmlNewNs(node, href, name);
	    if(nsPtr == NULL) {
#ifdef DEBUG_XMLSEC
    		xmlGenericError(xmlGenericErrorContext,
		    "xmlSecXPathTransformAdd: failed to add namespace\n");	
#endif
		xmlUnlinkNode(node);
		xmlFreeNode(node);
		return(-1);
	    }
	    p += 2;
	}
	if(defaultNsReplaced) {
	    xmlNsPtr ns;
	    /* the default namespace was set to xmlDSigNs but it 
	     * was overwritten here. let's try to set it back with 
	     * name="xmldsig" and hope for the best
	     * todo: probably there is a simple way to create a meaningfull 
	     * namespace
	     */	     
	    ns = xmlNewNs(NULL, xmlDSigNs, BAD_CAST "xmldsig");
	    if(ns == NULL) {
#ifdef DEBUG_XMLSEC
    		xmlGenericError(xmlGenericErrorContext,
		    "xmlSecXPathTransformAdd: failed to create \"xmldsig\" namespace\n");	
#endif
		xmlUnlinkNode(node);
		xmlFreeNode(node);
		return(-1);
	    }
	    node->ns = ns;
	}
    }
    return(0);
}

xmlNodeSetPtr		
xmlSecXPathTransformExecute (xmlDocPtr ctxDoc, xmlNodePtr ctxNode, 
			     xmlDocPtr doc, xmlNodeSetPtr nodes) {
    xmlNodePtr xpathNode;
    xmlXPathObjectPtr xpathObj;
    xmlNodeSetPtr ret;
    
    if((doc == NULL) || (ctxNode == NULL)){
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformExecute: doc is null\n");	
#endif
	return(NULL);
    }    
    
    /* find XPath children and read xpath expression */
    xpathNode = xmlSecGetNextElementNode(ctxNode->children);
    while((xpathNode != NULL) && !xmlSecCheckNodeName(doc, xpathNode, 
				         BAD_CAST "XPath", xmlDSigNs)) {
	xpathNode = xmlSecGetNextElementNode(xpathNode->next);
    }
    
    if(xpathNode == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext,
		"xmlSecXPathTransformExecute: node XPath is required for XPath transform\n");	
	return(NULL); 
#endif
    }
    
    /* function here() works only in he same document */
    xpathObj = xmlSecXPathTransformRead(doc, xpathNode, (doc == ctxDoc) ? xpathNode : NULL);
    if(xpathObj == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformExecute: xpathObj is null\n");	
#endif
	return(NULL);
    }
    
    if(nodes != NULL) {
	ret = xmlXPathIntersection(nodes, xpathObj->nodesetval);
    } else {
	ret = xmlXPathNodeSetMerge(NULL, xpathObj->nodesetval);
    }
    if(ret == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecXPathTransformExecute: ret is null\n");	
#endif
	xmlXPathFreeObject(xpathObj); 
	return(NULL);
    }   
    
    xmlXPathFreeObject(xpathObj); 
    return(ret);
}

/**
 * http://www.w3.org/TR/xmldsig-core/#sec-EnvelopedSignature
 *
 * An enveloped signature transform T removes the whole Signature element 
 * containing T from the digest calculation of the Reference element 
 * containing T. The entire string of characters used by an XML processor 
 * to match the Signature with the XML production element is removed. 
 * The output of the transform is equivalent to the output that would 
 * result from replacing T with an XPath transform containing the following 
 * XPath parameter element:
 *
 * <XPath xmlns:dsig="&dsig;">
 *   count(ancestor-or-self::dsig:Signature |
 *   here()/ancestor::dsig:Signature[1]) >
 *   count(ancestor-or-self::dsig:Signature)</XPath>
 *    
 * The input and output requirements of this transform are identical to 
 * those of the XPath transform, but may only be applied to a node-set from 
 * its parent XML document. Note that it is not necessary to use an XPath 
 * expression evaluator to create this transform. However, this transform 
 * MUST produce output in exactly the same manner as the XPath transform 
 * parameterized by the XPath expression above.
 *
 */
static xmlXPathObjectPtr
xmlSecEnvSigTransformPrepare(xmlDocPtr doc, xmlNodePtr ctxNode) {
    static const xmlChar expr[] = "(//. | //@* | //namespace::*)"
			       "[count(ancestor-or-self::dsig:Signature |"
			       "here()/ancestor::dsig:Signature[1]) >"
			       "count(ancestor-or-self::dsig:Signature)]";
       
    xmlXPathObjectPtr xpath; 
    xmlXPathContextPtr ctx; 
    
    if(doc == NULL) {   
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSigTransformPrepare: doc is null\n");	
#endif
	return(NULL);
    }

    ctx = xmlXPathNewContext(doc);
    if(ctx == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSigTransformPrepare: xpath context is null\n");	
#endif
	return(NULL);
    }

    xmlXPathRegisterFunc(ctx, BAD_CAST "here", xmlSecXPathHereFunction);
    ctx->here = ctxNode;
    ctx->xptr = 1;
    
    /*
     * Register namespace xmlDSig namespace
     */
    if(xmlXPathRegisterNs(ctx, BAD_CAST "dsig", xmlDSigNs) != 0) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext, 
	    "xmlSecEnvSigTransformPrepare: unable to register NS with prefix=\"dsig\"\n");
#endif
	xmlXPathFreeContext(ctx); 	     
	return(NULL);
    }

    /*  
     * Evaluate xpath
     */
    xpath = xmlXPathEvalExpression(expr, ctx);
    if(xpath == NULL) {
#ifdef DEBUG_XMLSEC
	xmlGenericError(xmlGenericErrorContext, 
	    "xmlSecEnvSigTransformPrepare: xpath eval failed\n");
#endif
	xmlXPathFreeContext(ctx); 
        return(NULL);
    }

    xmlXPathFreeContext(ctx);      
    return(xpath);
}

xmlNodeSetPtr
xmlSecEnvSignTransformExecute(xmlDocPtr ctxDoc, xmlNodePtr ctxNode,
			      xmlDocPtr doc, xmlNodeSetPtr nodes) {
    xmlXPathObjectPtr xpathObj;
    xmlNodeSetPtr ret;
    
    if((doc == NULL) || (ctxNode == NULL)){
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSignTransformExecute: doc or ctxNode is null\n");	
#endif
	return(NULL);
    }    
    
    if(doc != ctxDoc) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSignTransformExecute: EnvSign transform could be done in the same doc only\n");	
#endif
	return(NULL);
    }
    
    xpathObj = xmlSecEnvSigTransformPrepare(doc, ctxNode);
    if(xpathObj == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSignTransform: xpathObj is null\n");	
#endif
	return(NULL);
    }
    
    if(nodes != NULL) {
	ret = xmlXPathIntersection(nodes, xpathObj->nodesetval);
    } else {
	ret = xmlXPathNodeSetMerge(NULL, xpathObj->nodesetval);
    }
    if(ret == NULL) {
#ifdef DEBUG_XMLSEC
        xmlGenericError(xmlGenericErrorContext,
	    "xmlSecEnvSignTransform: ret is null\n");	
#endif
	xmlXPathFreeObject(xpathObj); 
	return(NULL);
    }   
    
    xmlXPathFreeObject(xpathObj); 
    return(ret);
}

