/** 
 * XML Security standards test: XMLDSig
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <openssl/err.h>
#include <openssl/rand.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <xmlsec/keysmgmt.h> 
#include <xmlsec/algorithms.h>
#include <xmlsec/hmac.h>
#include <xmlsec/dsa.h>
#include <xmlsec/xmldsig.h>


static const char usage[] = "<mode> [<options>] [<filename> [<filename> ... ]] \n"
			    "where <mode> is one of following\n"
			    "  sign        generate XML signature for the file\n"
			    "  verify      verify XML signature of the file\n"
			    "and <options> are from following list\n"
			    "  --keys <keys>    load keys from file <filename>\n"
			    "  --retry <number> repeat operation with each file <number> times\n" 
			    "  --debugDump <filenames-tmpl>\t dump to files before digesting or signing\n"
			    "";
int 
test_dsig(const char* filename, int generate, int retry);

xmlSecSimpleKeyMngrPtr keyMgr = NULL; 
clock_t total_time = 0;

const char *debugDumpTmpl = NULL;

int main(int argc, char **argv) {
    int keysLoaded = 0;    
    int ret = 0;
    int pos;
    int rnd_seed = 0;
    int retry = 1;
    
    while (RAND_status() != 1) {
	RAND_seed(&rnd_seed, sizeof(rnd_seed));
    }
    
    
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION


    keyMgr = xmlSecSimpleKeyMngrCreate();
    if(keyMgr == NULL) {
	fprintf(stderr, "Error: failed to create keys manager\n");
	goto done;	
    }
    
    /*
     * Parse command line and process file
     */
    if( argc < 3 ) {
	fprintf(stderr, "Error: wrong number of arguments.\nUsage: %s %s\n", argv[0], usage);
	goto done;		
    } 
    
    /* load options */
    pos = 2;
    while(pos < argc) {
	if(argv[pos][0] != '-') {
	    /* it's filename? */
	    break;
	} else if(strcmp(argv[pos], "--keys") == 0) {	  
	    if((++pos) >= argc) {
		fprintf(stderr, "Error: option \"%s\" requires argument.\nUsage: %s %s\n", argv[pos-1], argv[0], usage);
		goto done;
	    }
	    
	    ret = xmlSecSimpleKeyMngrLoad(keyMgr, argv[pos]);
	    if(ret < 0) {
		fprintf(stderr, "Error: failed to load keys from \"%s\".\nUsage: %s %s\n", argv[pos], argv[0], usage);
		goto done;
	    }
	    keysLoaded = 1;
	} else if(strcmp(argv[pos], "--retry") == 0) {
	    if(((++pos) >= argc) || (sscanf(argv[pos], "%d", &retry) != 1)) {
		fprintf(stderr, "Error: option \"%s\" requires number argument.\nUsage: %s %s\n", argv[pos-1], argv[0], usage);
		goto done;
	    }	    
	} else if(strcmp(argv[pos], "--debugDump") == 0) {
	    if((++pos) >= argc) {
		fprintf(stderr, "Error: option \"%s\" requires argument.\nUsage: %s %s\n", argv[pos-1], argv[0], usage);
		goto done;
	    }
	    debugDumpTmpl = argv[pos];	
	} else {
	    fprintf(stderr, "Error: bad option \"%s\".\nUsage: %s %s\n", argv[pos], argv[0], usage);
	    ret = -1;
	    goto done;
	}
	++pos;
    }
    
    while(pos < argc && ret >= 0) { 
	if(strcmp(argv[1], "sign") == 0) {
	    ret = test_dsig(argv[pos], 1, retry);
	} else if(strcmp(argv[1], "verify") == 0) {
	    ret = test_dsig(argv[pos], 0, retry);
	} else {
	    fprintf(stderr, "Error: bad mode \"%s\".\nUsage: %s %s\n", argv[1], argv[0], usage);
	    ret = -1;
	    goto done;
	}
	++pos;
    }

    fprintf(stderr, "Executed %d tests in %ld msec\n", retry, total_time / (CLOCKS_PER_SEC / 1000));

done:
    if(keyMgr != NULL) {
	xmlSecKeyMngrDestroy(keyMgr);
    }
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    xmlMemoryDump();
    

    RAND_cleanup();
    ERR_clear_error();

    return((ret >= 0) ? 0 : 1);
}

int 
test_dsig(const char* filename, int generate, int retry) {
    xmlDocPtr doc;
    int ret;
    int i;
    clock_t start_time;
    
    /*
     * build an XML tree from a the file; we need to add default
     * attributes and resolve all character and entities references
     */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    doc = xmlParseFile(filename);
    if (doc == NULL) {
	fprintf(stderr, "Error	: unable to parse file \"%s\"\n", filename);
	return(-1);
    }
    
    /*
     * Check the document is of the right kind
     */    
    if(xmlDocGetRootElement(doc) == NULL) {
        fprintf(stderr,"Error: empty document for file \"%s\"\n", filename);
	xmlFreeDoc(doc);
	return(-1);
    }

    start_time = clock();
    for(i = 0; i < retry; ++i) {
	xmlDSigCtxPtr dsigCtx;
	
	dsigCtx = xmlDSigCtxCreate(keyMgr, NULL);
	if(dsigCtx == NULL) {
    	    fprintf(stderr,"Error: empty document for file \"%s\"\n", filename);
	    xmlFreeDoc(doc);
	    return(-1);
	}
	
	/* add debug dump */
	dsigCtx->debugFileNamesTmpl = debugDumpTmpl;
	
	/*
	 * Generate/verify signature
	 */      
	if(generate) {
	    xmlChar *result;
	    int len;

	    fprintf(stderr, "Generating xml signature for \"%s\"\n", filename);	
	    ret = xmlDSigGenerate(dsigCtx, doc, NULL, NULL);
	    if(ret < 0) {
    		fprintf(stderr,"Error: xmlDSigGenerate() failed for file \"%s\"\n", filename);
		xmlDSigCtxDestroy(dsigCtx);
		xmlFreeDoc(doc);
		return(-1);
	    } 
	
	    /*
	     * Print document out in default UTF-8 encoding
	     */
	    xmlDocDumpMemoryEnc(doc, &result, &len, NULL);
	    if(result == NULL) {
		fprintf(stderr,"Error: failed to dump document to memory\n");
		xmlDSigCtxDestroy(dsigCtx);
		xmlFreeDoc(doc);  
		return(-1);
	    }
	    fwrite(result, len, 1, stdout);
	    xmlFree(result);
	} else {
	    fprintf(stderr, "Validating xml signature in \"%s\"\n", filename);	
	    ret = xmlDSigValidate(dsigCtx, doc, NULL, NULL);
	    if(ret < 0) {
		fprintf(stdout,"FAIL\n");
		xmlDSigCtxDestroy(dsigCtx);
		xmlFreeDoc(doc);                                                               
		return(-1);
	    } else {
		fprintf(stdout, "OK\n");
	    }
	}
	xmlDSigCtxDestroy(dsigCtx);
    }    
    
    total_time += clock() - start_time;
           
    /*
     * Cleanup
     */ 
    xmlFreeDoc(doc);    

    return(0);
}

