/** 
 * XML Security standards test: keys managemenet
 * 
 * See Copyright for the status of this software.
 * 
 * Author: Aleksey Sanin <aleksey@aleksey.com>
 */
#include <stdlib.h>
#include <string.h>

#include <openssl/rand.h>
#include <openssl/err.h>

#include <libxml/tree.h>

#include <xmlsec/keysmgmt.h>
#include <xmlsec/hmac.h>
#include <xmlsec/dsa.h>
#include <xmlsec/rsa.h>


static const char usage[] = "<mode> [<options>] <filename> \n"
			    "where <mode> is one of following\n"
			    "  privatekeys  \t to dump keys to a file\n"
			    "  publickeys   \t to dump keys to a file\n"
			    "  createkeys   \t to create keys to a file\n"
			    "and <options> are from following list\n"
			    "  --keys <keys>\t to load keys from file <filename>\n"
			    "";
int 
test_create_keys(const char* filename);

xmlSecSimpleKeyMngrPtr keyMgr = NULL;

int main(int argc, char **argv) {
    int keysLoaded = 0;    
    int ret = 0;
    int pos;
    int rnd_seed = 0;
    
    while (RAND_status() != 1) {
	RAND_seed(&rnd_seed, sizeof(rnd_seed));
    }

        
    /*
     * Init libxml
     */     
    xmlInitParser();
    LIBXML_TEST_VERSION
    
    keyMgr = xmlSecSimpleKeyMngrCreate();
    if(keyMgr == NULL) {
	fprintf(stderr, "Error: failed to create keys manager\n");
	goto done;	
    }
    
    /*
     * Parse command line and process file
     */
    if( argc < 3 ) {
	fprintf(stderr, "Error: wrong number of arguments.\nUsage: %s %s\n", argv[0], usage);
	goto done;		
    } 
    
    /* load options */
    pos = 2;
    while(pos < argc) {
	if(argv[pos][0] != '-') {
	    /* it's filename? */
	    break;
	} else if(strcmp(argv[pos], "--keys") == 0) {	  
	    /* load keys! */
	    if((++pos) >= argc) {
		fprintf(stderr, "Error: option \"%s\" requires argument.\nUsage: %s %s\n", argv[pos-1], argv[0], usage);
		goto done;
	    }
	    
	    ret = xmlSecSimpleKeyMngrLoad(keyMgr, argv[pos]);
	    if(ret < 0) {
		fprintf(stderr, "Error: failed to load keys from \"%s\".\nUsage: %s %s\n", argv[pos], argv[0], usage);
		goto done;
	    }
	    keysLoaded = 1;
	} else {
	    fprintf(stderr, "Error: bad option \"%s\".\nUsage: %s %s\n", argv[pos], argv[0], usage);
	    ret = -1;
	    goto done;
	}
	++pos;
    }
    
    while(pos < argc && ret >= 0) { 
	if(strcmp(argv[1], "privatekeys") == 0) {
	    if(!keysLoaded) {
		fprintf(stderr, "Error: keys not loaded.\nUsage: %s %s\n", argv[0], usage);
		goto done;
	    }
	    ret = xmlSecSimpleKeyMngrSave(keyMgr, argv[pos], 1);
	    if(ret < 0) {
		fprintf(stderr, "Error: \"%s\" failed\n", argv[1]);
	    }
	    goto done;
	} else if(strcmp(argv[1], "publickeys") == 0) {
	    if(!keysLoaded) {
		fprintf(stderr, "Error: keys not loaded.\nUsage: %s %s\n", argv[0], usage);
		goto done;
	    }
	    ret = xmlSecSimpleKeyMngrSave(keyMgr, argv[pos], 0);
	    if(ret < 0) {
		fprintf(stderr, "Error: \"%s\" failed\n", argv[1]);
	    }
	    goto done;
	} else if(strcmp(argv[1], "createkeys") == 0) {
	    ret = test_create_keys(argv[pos]);
	} else {
	    fprintf(stderr, "Error: bad mode \"%s\".\nUsage: %s %s\n", argv[1], argv[0], usage);
	    ret = -1;
	    goto done;
	}
	++pos;
    }


done:
    if(keyMgr != NULL) {
	xmlSecKeyMngrDestroy(keyMgr);
    }
    
    /* 
     * Shutdown libxml
     */
    xmlCleanupParser();
    xmlMemoryDump();
    
    RAND_cleanup();
    ERR_clear_error();

    return((ret >= 0) ? 0 : 1);
}


int 
test_create_keys(const char* filename) {
    static const char hmacKeyStr[] = "secret";
    int ret;
    xmlSecHmacKeyPtr hmacKey;
    xmlSecDsaKeyPtr dsaKey;
    xmlSecRsaKeyPtr rsaKey;
    int counter_ret;
    unsigned long h_ret;
    DSA *dsa;
    RSA *rsa;
      
    /* hmac */
    hmacKey = xmlSecHmacKeyCreate(NULL, BAD_CAST hmacKeyStr, strlen(hmacKeyStr));
    if(hmacKey == NULL) {
	fprintf(stderr, "Error: failed to create hmac key\n"); 
	return(-1);
    }    
    ret = xmlSecKeyMngrAddKey((xmlSecKeyMngrPtr)keyMgr, (xmlSecKeyPtr)hmacKey, NULL);
    if(ret < 0) {
	xmlSecKeyDestroy((xmlSecKeyPtr)hmacKey);
	fprintf(stderr, "Error: failed to add hmac key\n"); 
	return(-1);
    }

    /* dsa */
    dsaKey = xmlSecDsaKeyCreate(NULL);        
    if(dsaKey == NULL) {
	fprintf(stderr, "Error: failed to create dsa key\n"); 
	return(-1);
    }    
    dsa = DSA_generate_parameters(1024,NULL,0,&counter_ret,&h_ret,NULL,NULL);
    if(dsa == NULL) {
	xmlSecKeyDestroy((xmlSecKeyPtr)dsaKey);
	fprintf(stderr, "Error: failed to generate dsa key parameters\n"); 
	return(-1);
    }    
    ret = DSA_generate_key(dsa);
    if(ret < 0) {
	xmlSecKeyDestroy((xmlSecKeyPtr)dsaKey);
	fprintf(stderr, "Error: failed to generate dsa key\n"); 
	return(-1);
    }
    DSA_free(dsaKey->dsa);
    dsaKey->dsa = dsa;         
    ret = xmlSecKeyMngrAddKey((xmlSecKeyMngrPtr)keyMgr, (xmlSecKeyPtr)dsaKey, NULL);
    if(ret < 0) {
	xmlSecKeyDestroy((xmlSecKeyPtr)dsaKey);
	fprintf(stderr, "Error: failed to add dsa key\n"); 
	return(-1);
    }


    /* rsa */
    rsaKey = xmlSecRsaKeyCreate(NULL);        
    if(rsaKey == NULL) {
	fprintf(stderr, "Error: failed to create rsa key\n"); 
	return(-1);
    }    
    rsa = RSA_generate_key(1024, 3, NULL, NULL);
    if(rsa == NULL) {
	xmlSecKeyDestroy((xmlSecKeyPtr)rsaKey);
	fprintf(stderr, "Error: failed to generate rsa key\n"); 
	return(-1);
    }    
    RSA_free(rsaKey->rsa);
    rsaKey->rsa = rsa;         
    ret = xmlSecKeyMngrAddKey((xmlSecKeyMngrPtr)keyMgr, (xmlSecKeyPtr)rsaKey, NULL);
    if(ret < 0) {
	xmlSecKeyDestroy((xmlSecKeyPtr)rsaKey);
	fprintf(stderr, "Error: failed to add rsa key\n"); 
	return(-1);
    }

    ret = xmlSecSimpleKeyMngrSave(keyMgr,  filename, 1);
    if(ret < 0) {
	fprintf(stderr, "Error: failed to write to \"%s\"\n", filename);
	return(-1);	
    }
    return(0);
}    
